(() => {
    document.addEventListener('DOMContentLoaded', () => {
        if (document.getElementById('aipress-bubble')) return;

        // Generate or get session ID for conversation tracking
        let sessionId = sessionStorage.getItem('aipress_session_id');
        if (!sessionId) {
            sessionId = 'sess_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
            sessionStorage.setItem('aipress_session_id', sessionId);
        }

        /* 1. Bubble */
        const bubble = document.createElement('button');
        bubble.id = 'aipress-bubble';
        bubble.className = 'aipress-bubble';
        bubble.type = 'button';
        bubble.setAttribute('aria-label', 'Open chat');
        const chatIcon = aiPressCfg.branding?.chatIcon || `${aiPressCfg.url}assets/chat.png`;
        bubble.innerHTML = `<img src="${chatIcon}" alt="Chat">`;
        document.body.appendChild(bubble);

        /* Proactive Messages */
        let proactiveMessageShown = false;
        let proactiveTimer = null;
        
        const proactiveMessages = [
            "👋 Hi! Need any help?",
            "💬 Have a question? I'm here to help!",
            "🤔 Looking for something specific?",
            "✨ Hi there! Let me know if you need assistance!",
            "💡 Need help finding what you're looking for?",
            "😊 Hello! I'm here if you have any questions!",
            "🔍 Can I help you find anything?",
            "👍 Hi! Everything going okay?"
        ];
        
        function showProactiveMessage() {
            if (proactiveMessageShown || document.getElementById('aipress-panel')) {
                return; // Don't show if already shown or chat is open
            }
            
            const randomMessage = proactiveMessages[Math.floor(Math.random() * proactiveMessages.length)];
            
            const messageEl = document.createElement('div');
            messageEl.id = 'aipress-proactive-message';
            messageEl.className = 'aipress-proactive-message';
            messageEl.innerHTML = `
                <div class="aipress-proactive-content">
                    ${randomMessage}
                </div>
                <button class="aipress-proactive-close">×</button>
            `;
            
            document.body.appendChild(messageEl);
            
            // Animate in
            setTimeout(() => {
                messageEl.classList.add('show');
            }, 100);
            
            // Auto-hide after 8 seconds
            setTimeout(() => {
                hideProactiveMessage();
            }, 8000);
            
            // Close button
            messageEl.querySelector('.aipress-proactive-close').addEventListener('click', (e) => {
                e.stopPropagation();
                hideProactiveMessage();
                proactiveMessageShown = true; // Don't show again this session
            });
            
            // Click to open chat
            messageEl.addEventListener('click', () => {
                hideProactiveMessage();
                openPanel();
            });
        }
        
        function hideProactiveMessage() {
            const messageEl = document.getElementById('aipress-proactive-message');
            if (messageEl) {
                messageEl.classList.add('hide');
                setTimeout(() => {
                    messageEl.remove();
                }, 300);
            }
        }
        
        // Start proactive message timer
        function startProactiveTimer() {
            if (proactiveTimer) return;
            
            // First message after 30 seconds
            proactiveTimer = setTimeout(() => {
                showProactiveMessage();
                
                // Subsequent messages every 2 minutes if not shown yet
                proactiveTimer = setInterval(() => {
                    if (!proactiveMessageShown && !document.getElementById('aipress-panel')) {
                        showProactiveMessage();
                    }
                }, 120000); // 2 minutes
            }, 30000); // 30 seconds
        }
        
        function stopProactiveTimer() {
            if (proactiveTimer) {
                clearTimeout(proactiveTimer);
                clearInterval(proactiveTimer);
                proactiveTimer = null;
            }
            hideProactiveMessage();
        }
        
        // Start the timer when page loads
        startProactiveTimer();
        
        // Stop timer when chat is opened
        const originalOpenPanel = openPanel;
        openPanel = function() {
            stopProactiveTimer();
            proactiveMessageShown = true;
            originalOpenPanel();
        };
        
        /* Restore last state */
        if (localStorage.getItem('aipress_open') === '1') {
            openPanel();
        }

        bubble.addEventListener('click', () => {
            const panel = document.getElementById('aipress-panel');
            if (panel && panel.style.display !== 'none') {
                closePanel();
            } else {
                openPanel();
            }
        });

        function openPanel() {
            let panel = document.getElementById('aipress-panel');
            if (!panel) buildPanel();
            panel.style.display = 'flex';
            localStorage.setItem('aipress_open', '1');
            panel.querySelector('.aipress-input').focus();
        }

        function closePanel() {
            const panel = document.getElementById('aipress-panel');
            if (panel) panel.remove();
            localStorage.setItem('aipress_open', '0');
            bubble.focus();
        }

        function addMessage(content, isUser = false) {
            const log = document.querySelector('.aipress-log');
            const messageDiv = document.createElement('div');
            messageDiv.className = `aipress-message ${isUser ? 'user' : 'bot'}`;
            
            const avatar = document.createElement('div');
            avatar.className = 'aipress-avatar';
            if (isUser) {
                avatar.innerHTML = '👤';
            } else {
                const avatarUrl = aiPressCfg.branding?.avatar || `${aiPressCfg.url}assets/logo-aipress.png`;
                avatar.innerHTML = `<img src="${avatarUrl}" alt="AI">`;
            }
            
            const bubble = document.createElement('div');
            bubble.className = 'aipress-bubble-msg';
            bubble.innerHTML = content;
            
            messageDiv.appendChild(avatar);
            messageDiv.appendChild(bubble);
            log.appendChild(messageDiv);
            
            return { messageDiv, bubble };
        }
        
        // Add these functions before the buildPanel() function

        function saveChatHistory() {
            const log = document.querySelector('.aipress-log');
            if (log) {
                const messages = Array.from(log.querySelectorAll('.aipress-message')).map(msg => {
                    const isUser = msg.classList.contains('user');
                    const content = msg.querySelector('.aipress-bubble-msg').innerHTML;
                    return { isUser, content };
                });
                localStorage.setItem('aipress_chat_history', JSON.stringify(messages));
            }
        }
        
        function loadChatHistory() {
            const saved = localStorage.getItem('aipress_chat_history');
            if (saved) {
                try {
                    const messages = JSON.parse(saved);
                    const log = document.querySelector('.aipress-log');
                    
                    // Clear any existing welcome message
                    log.innerHTML = '';
                    
                    // Restore all messages
                    messages.forEach(msg => {
                        const { messageDiv } = addMessage(msg.content, msg.isUser);
                    });
                    
                    // Scroll to bottom
                    setTimeout(() => {
                        log.scrollTop = log.scrollHeight;
                    }, 100);
                    
                    return messages.length > 0;
                } catch (e) {
                    console.warn('Failed to load chat history:', e);
                }
            }
            return false;
        }
        
        function clearChatHistory() {
            localStorage.removeItem('aipress_chat_history');
        }

        function typewriterEffect(element, text, speed = 5) {
            return new Promise((resolve) => {
                const log = document.querySelector('.aipress-log');
                const messageContainer = element.closest('.aipress-message');
                
                // Pre-render the full text invisibly to calculate height
                const tempDiv = document.createElement('div');
                tempDiv.style.visibility = 'hidden';
                tempDiv.style.position = 'absolute';
                tempDiv.style.width = element.offsetWidth + 'px';
                tempDiv.style.whiteSpace = 'pre-wrap';
                tempDiv.style.wordWrap = 'break-word';
                tempDiv.className = element.className;
                tempDiv.innerHTML = text;
                document.body.appendChild(tempDiv);
                
                const finalHeight = tempDiv.offsetHeight;
                document.body.removeChild(tempDiv);
                
                // Set the element to its final height immediately
                element.style.minHeight = finalHeight + 'px';
                element.innerHTML = '';
                
                // Scroll to show the message from the top
                const messageTop = messageContainer.offsetTop - log.offsetTop;
                log.scrollTo({
                    top: messageTop - 10,
                    behavior: 'smooth'
                });
                
                let i = 0;
                function typeChar() {
                    if (i < text.length) {
                        element.innerHTML += text.charAt(i);
                        i++;
                        setTimeout(typeChar, speed);
                    } else {
                        // Clear the min-height after typing is complete
                        element.style.minHeight = '';
                        resolve();
                    }
                }
                
                typeChar();
            });
        }

        function scrollToMessage(messageElement) {
            const log = document.querySelector('.aipress-log');
            const messageRect = messageElement.getBoundingClientRect();
            const logRect = log.getBoundingClientRect();
            
            // Scroll to show the message from the top
            const scrollOffset = messageElement.offsetTop - log.offsetTop - 10;
            log.scrollTo({
                top: scrollOffset,
                behavior: 'smooth'
            });
        }

        function buildPanel() {
            const title = aiPressCfg.branding?.title || 'aiPress Chat';
            const avatar = aiPressCfg.branding?.avatar || `${aiPressCfg.url}assets/logo-aipress.png`;
            const hideAiPress = aiPressCfg.branding?.hideAiPressBranding || false;
            const panel = document.createElement('div');
            panel.id = 'aipress-panel';
            panel.className = 'aipress-panel';
            panel.setAttribute('role', 'dialog');
            panel.setAttribute('aria-modal', 'true');
            panel.innerHTML = `
                <div class="aipress-header">
                    <img src="${avatar}" alt="${title} logo">
                    <span class="aipress-title">${title}</span>
                    <button class="aipress-close" aria-label="Close chat">×</button>
                </div>
                <div class="aipress-log" tabindex="0" aria-live="polite"></div>
                <div class="aipress-input-container">
                    <textarea class="aipress-input" rows="1" placeholder="Ask me anything..."></textarea>
                    <button class="aipress-send-btn" aria-label="Send message">
                        <svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor">
                            <path d="M2.01 21L23 12 2.01 3 2 10l15 2-15 2z"/>
                        </svg>
                    </button>
                </div>
                ${hideAiPress ? '' : '<div class="aipress-powered">Powered by <a href="https://aipresschat.com" target="_blank">aiPress Chat</a></div>'}
            `;
            if (aiPressCfg.branding?.primaryColor && aiPressCfg.branding?.secondaryColor) {
                const style = document.createElement('style');
                style.textContent = `
                    .aipress-header {
                        background: linear-gradient(135deg, ${aiPressCfg.branding.primaryColor} 0%, ${aiPressCfg.branding.secondaryColor} 100%) !important;
                    }
                    .aipress-message.user .aipress-bubble-msg {
                        background: linear-gradient(135deg, ${aiPressCfg.branding.primaryColor} 0%, ${aiPressCfg.branding.secondaryColor} 100%) !important;
                    }
                    .aipress-send-btn {
                        background: linear-gradient(135deg, ${aiPressCfg.branding.primaryColor} 0%, ${aiPressCfg.branding.secondaryColor} 100%) !important;
                    }
                    .aipress-input:focus {
                        border-color: ${aiPressCfg.branding.primaryColor} !important;
                        box-shadow: 0 0 0 3px ${aiPressCfg.branding.primaryColor}20 !important;
                    }
                `;
                document.head.appendChild(style);
            }
            document.body.appendChild(panel);

            const closeBtn = panel.querySelector('.aipress-close');
            const log = panel.querySelector('.aipress-log');
            const input = panel.querySelector('.aipress-input');
            const sendBtn = panel.querySelector('.aipress-send-btn');

            setTimeout(() => {
                const hasHistory = loadChatHistory();
                if (!hasHistory) {
                    const { messageDiv, bubble } = addMessage('👋 Hi! I\'m here to help you with any questions about aiPress Chat. What would you like to know?', false);
                    scrollToMessage(messageDiv);
                }
            }, 300);

            /* Close actions */
            closeBtn.addEventListener('click', closePanel);
            panel.addEventListener('keydown', (e) => {
                if (e.key === 'Escape') {
                    e.preventDefault();
                    closePanel();
                }
            });

            /* Focus trap */
            panel.addEventListener('keydown', (e) => {
                if (e.key !== 'Tab') return;
                const focusable = panel.querySelectorAll('button, textarea');
                const first = focusable[0];
                const last = focusable[focusable.length - 1];
                if (e.shiftKey && document.activeElement === first) {
                    e.preventDefault();
                    last.focus();
                } else if (!e.shiftKey && document.activeElement === last) {
                    e.preventDefault();
                    first.focus();
                }
            });

            /* Auto-grow textarea */
            input.addEventListener('input', () => {
                input.style.height = 'auto';
                input.style.height = Math.min(input.scrollHeight, 100) + 'px';
            });

            /* Send message function */
            const sendMessage = async () => {
                const q = input.value.trim();
                if (!q) return;
                
                // Disable input during processing
                input.disabled = true;
                sendBtn.disabled = true;
                
                // Add user message
                const userMessage = addMessage(q, true);
                input.value = '';
                input.style.height = 'auto';
                scrollToMessage(userMessage.messageDiv);

                // Add typing indicator
                const typingDiv = document.createElement('div');
                typingDiv.className = 'aipress-message bot typing-indicator';
                typingDiv.innerHTML = `
                    <div class="aipress-avatar">
                        <img src="${aiPressCfg.branding?.avatar || `${aiPressCfg.url}assets/logo-aipress.png`}" alt="AI">
                    </div>
                    <div class="aipress-bubble-msg">
                        <div class="aipress-typing">
                            <span></span><span></span><span></span>
                        </div>
                    </div>
                `;
                log.appendChild(typingDiv);
                scrollToMessage(typingDiv);

                // Track request start time for response time logging
                const startTime = Date.now();

                const fd = new FormData();
                fd.append('action', 'aipress_chat');
                fd.append('question', q);
                fd.append('nonce', aiPressCfg.nonce);
                fd.append('page_id', aiPressCfg.page_id);
                fd.append('session_id', sessionId);

                const res = await fetch(aiPressCfg.ajax, {
                        method: 'POST',
                        body: fd,
                        headers: { 'X-WP-Nonce': aiPressCfg.nonce },
                    })
                    .then((r) => r.json())
                    .catch(() => ({ data: { answer: 'Error connecting to chat service. Please try again.' } }));

                // Calculate response time
                const responseTime = Date.now() - startTime;

                // Remove typing indicator
                typingDiv.remove();

                // Enhanced error handling
                let answer = 'Sorry, I encountered an error. Please try again.';
                if (res && res.data && res.data.answer) {
                    answer = res.data.answer;
                } else if (res && res.success === false && res.data) {
                    answer = 'Error: ' + res.data;
                }

                // Add bot response with typewriter effect
                const { messageDiv, bubble } = addMessage('', false);
                scrollToMessage(messageDiv);
                
                // Type out the response
                await typewriterEffect(bubble, answer, 1);
                saveChatHistory();

                // Re-enable input
                input.disabled = false;
                sendBtn.disabled = false;
                input.focus();

                // Debug info (remove in production)
                console.log('aiPress Debug:', {
                    sessionId: sessionId,
                    responseTime: responseTime + 'ms',
                    pageId: aiPressCfg.page_id
                });
            };

            /* Submit on Enter */
            input.addEventListener('keydown', async (e) => {
                if (e.key === 'Enter' && !e.shiftKey) {
                    e.preventDefault();
                    await sendMessage();
                }
            });

            /* Send button click */
            sendBtn.addEventListener('click', sendMessage);
        }
    });
})();