<?php
/**
 * Conversation Logs Admin Page
 */

namespace AIPress\Admin;

/**
 * Conversation Logs admin page.
 */
class ConversationLogs {
    
    /**
     * Constructor.
     */
    public function __construct() {
        add_action( 'admin_menu', array( $this, 'add_menu' ) );
        add_action( 'wp_ajax_aipress_delete_conversation', array( $this, 'delete_conversation' ) );
    }
    
    /**
     * Add admin menu.
     */
    public function add_menu() {
        // Only show for pro users
        if ( ! aipress_is_pro() ) {
            return;
        }
        
        add_submenu_page(
            'options-general.php',
            'aiPress Chat Logs',
            'aiPress Logs',
            'manage_options',
            'aipress-logs',
            array( $this, 'logs_page' )
        );
    }
    
    /**
     * Display the logs page.
     */
    public function logs_page() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'aipress_logs';
        
        wp_enqueue_style('aipress-admin', AIPRESS_URL . 'assets/admin-styles.css', array(), AIPRESS_VERSION);
        wp_enqueue_script('aipress-admin-js', AIPRESS_URL . 'assets/admin.js', array('jquery'), AIPRESS_VERSION, true);
        
        // Check if table exists
        if ( $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table_name ) ) !== $table_name ) {
            ?>
            <div class="wrap">
                <h1>aiPress Chat Logs</h1>
                <div class="notice notice-warning">
                    <p>Conversation logs table not found. Please ensure the plugin is properly activated.</p>
                </div>
            </div>
            <?php
            return;
        }
        
        // Handle bulk actions
        if ( isset( $_POST['action'] ) && $_POST['action'] === 'delete_selected' && isset( $_POST['conversation_ids'] ) && wp_verify_nonce( $_POST['aipress_nonce'] ?? '', 'aipress_bulk_delete' ) ) {
            $this->handle_bulk_delete();
        }
        
        // Pagination
        $per_page = 20;
        $current_page = isset( $_GET['paged'] ) ? max( 1, intval( sanitize_text_field( wp_unslash( $_GET['paged'] ) ) ) ) : 1;
        $offset = ( $current_page - 1 ) * $per_page;
        
        // Get total count
        $total_items = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM %i", $table_name ) );
        $total_pages = ceil( $total_items / $per_page );
        
        // Get conversations
        $conversations = $wpdb->get_results( $wpdb->prepare(
            "SELECT * FROM %i ORDER BY created_at DESC LIMIT %d OFFSET %d",
            $table_name,
            $per_page,
            $offset
        ) );
        
        // Get stats
        $stats = aipress_get_conversation_stats();
        
        ?>
        <div class="wrap">
            <!-- Header -->
            <div style="display:flex; align-items:center; justify-content:space-between; margin-bottom:20px;">
                <div style="display:flex; align-items:center;">
                    <img src="<?php echo esc_url( AIPRESS_URL . 'assets/logo-aipress.png' ); ?>" alt="aiPress Chat" style="height:48px; margin-right:12px;" />
                    <h1 style="margin:0;">
                        aiPress Chat Logs
                        <span style="background: #46b450; color: white; padding: 2px 8px; border-radius: 4px; font-size: 12px; margin-left: 8px;">PRO</span>
                    </h1>
                </div>
                <a href="<?php echo esc_url(admin_url( 'options-general.php?page=aipress-chat' )); ?>" class="button">← Back to Settings</a>
            </div>
            
            <!-- Stats Cards -->
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; margin-bottom: 25px;">
                <div style="background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
                    <h3 style="margin: 0 0 10px 0; color: #0073aa;">Total Conversations</h3>
                    <p style="font-size: 24px; font-weight: bold; margin: 0;"><?php echo esc_html( $stats['total_conversations'] ?? 0 ); ?></p>
                </div>
                <div style="background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
                    <h3 style="margin: 0 0 10px 0; color: #46b450;">Today</h3>
                    <p style="font-size: 24px; font-weight: bold; margin: 0;"><?php echo esc_html( $stats['today'] ?? 0 ); ?></p>
                </div>
                <div style="background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
                    <h3 style="margin: 0 0 10px 0; color: #f56565;">This Week</h3>
                    <p style="font-size: 24px; font-weight: bold; margin: 0;"><?php echo esc_html( $stats['this_week'] ?? 0 ); ?></p>
                </div>
                <div style="background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
                    <h3 style="margin: 0 0 10px 0; color: #8b5cf6;">Unique Sessions</h3>
                    <p style="font-size: 24px; font-weight: bold; margin: 0;"><?php echo esc_html( $stats['unique_sessions'] ?? 0 ); ?></p>
                </div>
            </div>
            
            <?php if ( empty( $conversations ) ) : ?>
                <div style="background: white; padding: 40px; text-align: center; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
                    <h3>No conversations yet</h3>
                    <p>Once visitors start chatting with your AI assistant, conversations will appear here.</p>
                    <p><a href="<?php echo esc_url(home_url()); ?>" class="button button-primary" target="_blank">Visit Your Site →</a></p>
                </div>
            <?php else : ?>
                <!-- Conversations Table -->
                <form method="post">
                    <?php wp_nonce_field( 'aipress_bulk_delete', 'aipress_nonce' ); ?>
                    
                    <div class="tablenav top">
                        <div class="alignleft actions">
                            <select name="action">
                                <option value="">Bulk Actions</option>
                                <option value="delete_selected">Delete Selected</option>
                            </select>
                            <input type="submit" class="button" value="Apply">
                        </div>
                        
                        <?php if ( $total_pages > 1 ) : ?>
                            <div class="tablenav-pages">
                                <span class="displaying-num"><?php echo esc_attr($total_items); ?> items</span>
                                <?php
                                $pagination = paginate_links( array(
                                    'base' => add_query_arg( 'paged', '%#%' ),
                                    'format' => '',
                                    'prev_text' => '&laquo;',
                                    'next_text' => '&raquo;',
                                    'total' => $total_pages,
                                    'current' => $current_page,
                                    'type' => 'plain'
                                ) );
                                echo esc_attr($pagination);
                                ?>
                            </div>
                        <?php endif; ?>
                    </div>
                    
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <td class="manage-column column-cb check-column">
                                    <input type="checkbox" id="cb-select-all">
                                </td>
                                <th>Time</th>
                                <th>Session</th>
                                <th>User Message</th>
                                <th>AI Response</th>
                                <th>Page</th>
                                <th>Model</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ( $conversations as $conversation ) : ?>
                                <tr>
                                    <th class="check-column">
                                        <input type="checkbox" name="conversation_ids[]" value="<?php echo esc_attr( $conversation->id ); ?>">
                                    </th>
                                    <td>
                                        <a href="#" 
                                           class="button button-small view-chat-btn" 
                                           data-conversation-id="<?php echo intval($conversation->id); ?>"
                                           data-conversation-date="<?php echo esc_attr(date('M j, Y g:i A', strtotime($conversation->created_at))); ?>">
                                            View Chat
                                        </a>
                                        <strong><?php echo esc_html( gmdate( 'M j, g:i A', strtotime( $conversation->created_at ) ) ); ?></strong><br>
                                        <small style="color: #666;"><?php echo esc_html( human_time_diff( strtotime( $conversation->created_at ) ) ); ?> ago</small>
                                    </td>
                                    <td>
                                        <code style="font-size: 11px;"><?php echo esc_html( substr( $conversation->session_id, 0, 12 ) ); ?>...</code><br>
                                        <small style="color: #666;"><?php echo esc_html( $conversation->user_ip ); ?></small>
                                    </td>
                                    <td style="max-width: 250px;">
                                        <div style="max-height: 60px; overflow: hidden; line-height: 1.4;">
                                            <?php echo esc_html( wp_trim_words( $conversation->user_message, 15 ) ); ?>
                                        </div>
                                    </td>
                                    <td style="max-width: 300px;">
                                        <div style="max-height: 60px; overflow: hidden; line-height: 1.4;">
                                            <?php echo esc_html( wp_trim_words( $conversation->ai_response, 20 ) ); ?>
                                        </div>
                                    </td>
                                    <td>
                                        <?php if ( $conversation->page_id ) : ?>
                                            <?php $page_title = get_the_title( $conversation->page_id ); ?>
                                            <?php if ( $page_title ) : ?>
                                                <a href="<?php echo esc_url( get_permalink( $conversation->page_id ) ); ?>" target="_blank" title="<?php echo esc_attr( $page_title ); ?>">
                                                    <?php echo esc_html( wp_trim_words( $page_title, 3 ) ); ?>
                                                </a>
                                            <?php else : ?>
                                                <span style="color: #999;">Page ID <?php echo esc_html( $conversation->page_id ); ?></span>
                                            <?php endif; ?>
                                        <?php else : ?>
                                            <span style="color: #999;">Unknown</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <small><?php echo esc_html( $conversation->model ?: 'Unknown' ); ?></small>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                    
                    <?php if ( $total_pages > 1 ) : ?>
                        <div class="tablenav bottom">
                            <div class="tablenav-pages">
                                <?php echo esc_attr($pagination); ?>
                            </div>
                        </div>
                    <?php endif; ?>
                </form>
            <?php endif; ?>
        </div>
        
        <script>
        // Select all checkbox functionality
        document.getElementById('cb-select-all')?.addEventListener('change', function() {
            const checkboxes = document.querySelectorAll('input[name="conversation_ids[]"]');
            checkboxes.forEach(cb => cb.checked = this.checked);
        });
        </script>
        <?php
    }
    
    /**
     * Handle bulk delete action.
     */
    private function handle_bulk_delete() {
        if ( ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['aipress_nonce'] ?? '' ) ), 'aipress_bulk_delete' ) ) {
            wp_die( 'Security check failed' );
        }
        
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( 'Insufficient permissions' );
        }
        
        $conversation_ids = isset( $_POST['conversation_ids'] ) ? array_map( 'intval', wp_unslash( $_POST['conversation_ids'] ) ) : array();
        
        if ( empty( $conversation_ids ) ) {
            return;
        }
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'aipress_logs';
        $placeholders = implode( ',', array_fill( 0, count( $conversation_ids ), '%d' ) );
        
        $deleted = $wpdb->query( $wpdb->prepare( "DELETE FROM %i WHERE id IN ($placeholders)", $table_name,
            $conversation_ids
        ) );
        
        if ( $deleted ) {
            add_action( 'admin_notices', function() use ( $deleted ) {
                echo '<div class="notice notice-success"><p>' . sprintf( 'Deleted %d conversation(s).', esc_attr($deleted) ) . '</p></div>';
            } );
        }
    }
    
    /**
     * AJAX handler for deleting single conversation.
     */
    public function delete_conversation() {
        if ( ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ?? '' ) ), 'aipress_delete_conversation' ) ) {
            wp_send_json_error( 'Security check failed' );
        }
        
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }
        
        $conversation_id = isset( $_POST['conversation_id'] ) ? intval( sanitize_text_field( wp_unslash( $_POST['conversation_id'] ) ) ) : 0;
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'aipress_logs';
        
        $deleted = $wpdb->delete( $table_name, array( 'id' => $conversation_id ), array( '%d' ) );
        
        if ( $deleted ) {
            wp_send_json_success( 'Conversation deleted' );
        } else {
            wp_send_json_error( 'Failed to delete conversation' );
        }
    }
}