<?php
/**
 * Meta Box for Custom aiPress Content (Pro Only)
 *
 * @package aiPress_Chat
 */

namespace AIPress\Admin;

/**
 * Meta Box component for adding custom content to pages/posts (Pro feature).
 */
class MetaBox {
    
    /**
     * Constructor.
     */
    public function __construct() {
        // Only initialize if Pro
        if ( ! aipress_is_pro() ) {
            return;
        }
        
        add_action( 'add_meta_boxes', array( $this, 'add_meta_box' ) );
        add_action( 'save_post', array( $this, 'save_meta_box' ) );
    }
    
    /**
     * Add meta box to post/page edit screen.
     */
    public function add_meta_box() {
        // Double-check Pro status
        if ( ! aipress_is_pro() ) {
            return;
        }
        
        // Get all public post types for Pro users
        $post_types = get_post_types( array( 'public' => true ), 'names' );
        
        foreach ( $post_types as $post_type ) {
            add_meta_box(
                'aipress_content_meta_box',
                'aiPress Chat Content (Pro)',
                array( $this, 'meta_box_callback' ),
                $post_type,
                'normal',
                'default'
            );
        }
    }
    
    /**
     * Meta box display callback.
     *
     * @param \WP_Post $post The post object.
     */
    public function meta_box_callback( $post ) {
        // Final Pro check
        if ( ! aipress_is_pro() ) {
            echo '<div style="padding: 15px; background: #fff3cd; border-left: 4px solid #ffc107;">';
            echo '<strong>🔒 Pro Feature</strong><br>';
            echo 'Custom content fields are available with aiPress Chat Pro. ';
            echo '<a href="https://aipresschat.com/pricing/" target="_blank">Upgrade to Pro →</a>';
            echo '</div>';
            return;
        }
        
        // Add nonce for security
        wp_nonce_field( 'aipress_meta_box', 'aipress_meta_box_nonce' );
        
        // Get current values
        $content = get_post_meta( $post->ID, '_aipress_content', true );
        $mode = get_post_meta( $post->ID, '_aipress_content_mode', true ) ?: 'replace';
        
        ?>
        <div class="aipress-meta-box">
            <p><strong>Add content for the AI chatbot that isn't visible on your page.</strong></p>
            <p>This Pro feature is useful for:</p>
            <ul style="margin-left: 20px;">
                <li>• FAQ content that doesn't fit your page design</li>
                <li>• Technical details for customer support</li>
                <li>• Additional context about your products/services</li>
                <li>• Page builders where content isn't detected automatically</li>
            </ul>
            
            <table class="form-table" style="margin-top: 15px;">
                <tr>
                    <th scope="row">
                        <label for="aipress_content_mode">Content Mode:</label>
                    </th>
                    <td>
                        <select id="aipress_content_mode" name="aipress_content_mode" style="width: 200px;">
                            <option value="replace" <?php selected( $mode, 'replace' ); ?>>Replace page content</option>
                            <option value="add" <?php selected( $mode, 'add' ); ?>>Add to page content</option>
                        </select>
                        <p class="description">
                            <strong>Replace:</strong> Only use this custom content for AI responses<br>
                            <strong>Add:</strong> Combine this with the page's existing content
                        </p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="aipress_content">Custom Content:</label>
                    </th>
                    <td>
                        <textarea 
                            id="aipress_content" 
                            name="aipress_content" 
                            rows="8" 
                            style="width: 100%; max-width: 600px;"
                            placeholder="Enter additional content for the AI chatbot to reference when answering questions about this page..."
                        ><?php echo esc_textarea( $content ); ?></textarea>
                        <p class="description">
                            This content will be indexed by aiPress Chat but won't appear on your page.
                        </p>
                    </td>
                </tr>
            </table>
            
            <?php if ( ! empty( $content ) ): ?>
                <div style="margin-top: 15px; padding: 10px; background: #f0f8ff; border-left: 4px solid #0073aa;">
                    <strong>📝 Content Status:</strong> This page has custom aiPress content (<?php echo strlen( $content ); ?> characters)
                </div>
            <?php endif; ?>
        </div>
        
        <style>
        .aipress-meta-box ul {
            list-style: none;
            padding: 0;
        }
        .aipress-meta-box li {
            margin-bottom: 5px;
            color: #666;
        }
        </style>
        <?php
    }
    
    /**
     * Save meta box data.
     *
     * @param int $post_id The post ID.
     */
    public function save_meta_box( $post_id ) {
        // Only save for Pro users
        if ( ! aipress_is_pro() ) {
            return;
        }
        
        // Check nonce
        if ( ! isset( $_POST['aipress_meta_box_nonce'] ) || 
             ! wp_verify_nonce( $_POST['aipress_meta_box_nonce'], 'aipress_meta_box' ) ) {
            return;
        }
        
        // Check user permissions
        if ( ! current_user_can( 'edit_post', $post_id ) ) {
            return;
        }
        
        // Don't save on autosave
        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
            return;
        }
        
        // Save the content
        if ( isset( $_POST['aipress_content'] ) ) {
            $content = sanitize_textarea_field( wp_unslash( $_POST['aipress_content'] ) );
            update_post_meta( $post_id, '_aipress_content', $content );
        }
        
        // Save the mode
        if ( isset( $_POST['aipress_content_mode'] ) ) {
            $mode = sanitize_text_field( wp_unslash( $_POST['aipress_content_mode'] ) );
            if ( in_array( $mode, array( 'replace', 'add' ), true ) ) {
                update_post_meta( $post_id, '_aipress_content_mode', $mode );
            }
        }
    }
}