<?php
/**
 * AiPress Chat — Chat component.
 *
 * @package aiPress_Chat
 */

namespace AIPress;

/**
 * Chat component.
 */
class Chat {
	/**
	 * Your API key, from settings or wp-config.
	 *
	 * @var string
	 */
	private string $key;
	
	/**
	 * Constructor.
	 *
	 * Hooks into WP to enqueue assets and register AJAX.
	 */
	public function __construct() {
		add_action( 'wp_enqueue_scripts', array( $this, 'assets' ) );
		add_action( 'wp_ajax_nopriv_aipress_chat', array( $this, 'ajax' ) );
		add_action( 'wp_ajax_aipress_chat', array( $this, 'ajax' ) );
		$this->key = get_option( 'aipress_settings' )['openai_key'] ?? ( defined( 'AIPRESS_OPENAI_KEY' ) ? AIPRESS_OPENAI_KEY : '' );
	}
	
	/**
	 * Enqueue front-end JS/CSS and localize.
	 *
	 * @return void
	 */
	public function assets() {
		wp_enqueue_style( 'aipress-css', AIPRESS_URL . '/assets/frontend.css', array(), AIPRESS_VERSION );
		wp_enqueue_script( 'aipress-js', AIPRESS_URL . '/assets/frontend.js', array(), AIPRESS_VERSION, true );
		
		$settings = get_option( 'aipress_settings', array() );
        $branding = array();
        
        if ( aipress_is_pro() ) {
            $branding = array(
                'title' => $settings['chat_title'] ?? 'AI Assistant',
                'avatar' => $settings['chat_avatar'] ?? '',
                'chatIcon' => $settings['chat_icon'] ?? '',
                'primaryColor' => $settings['primary_color'] ?? '#667eea',
                'secondaryColor' => $settings['secondary_color'] ?? '#764ba2',
                'hideAiPressBranding' => !empty( $settings['hide_aipress_branding'] )
            );
        }
		
		wp_localize_script(
			'aipress-js',
			'aiPressCfg',
			array(
				'ajax'  => admin_url( 'admin-ajax.php' ),
				'nonce' => wp_create_nonce( 'aipress_chat' ),
				'url'   => AIPRESS_URL,
				'page_id' => get_queried_object_id(),
				'branding' => $branding
			)
		);
	}
	
	/**
	 * Spam guard component.
	 *
	 * Enforces a 2-second per-IP cooldown and a 100-calls-per-hour bucket.
	 *
	 * @throws \Exception If the request is too frequent or hourly limit exceeded.
	 *
	 * @return void
	 */
	private function spam_guard() {
		$ip = sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ?? 'x' ) );
		
		// Check 2-second cooldown
		if ( get_transient( 'aipress_' . $ip . '_2s' ) ) {
			throw new \Exception( '⚠️ Please wait a moment before sending another message.' );
		}
		set_transient( 'aipress_' . $ip . '_2s', 1, 2 );
		
		// Check hourly limit
		$bucket = 'aipress_' . md5( $ip . '_' . gmdate( 'YmdH' ) );
		$cnt    = (int) get_transient( $bucket );
		if ( $cnt >= 100 ) {
			throw new \Exception( '⚠️ You\'ve reached the hourly message limit. Please try again later.' );
		}
		set_transient( $bucket, $cnt + 1, 3600 );
	}
	
	/**
	 * AJAX handler: sanitize, embed, retrieve, call OpenAI, return JSON.
	 *
	 * @return void
	 */
	public function ajax() {
		try {
			check_ajax_referer( 'aipress_chat', 'nonce', true );
			
			// Use the spam_guard method
			$this->spam_guard();

			$q = sanitize_text_field( wp_unslash( $_POST['question'] ?? '' ) );
			if ( ! $q ) {
				wp_send_json_success( array( 'answer' => 'Hi there! What can I help you with today?' ) );
			}
			if ( ! $this->key ) {
				wp_send_json_success( array( 'answer' => 'Sorry, the chat service is currently unavailable. Please try again later or contact support.' ) );
			}

			$embed_q = $this->embed( $q );
			if ( ! $embed_q ) {
				wp_send_json_success( array( 'answer' => 'I\'m having trouble processing your message right now. Please try again.' ) );
			}
			
			global $wpdb;
			
			// Get all indexed content (remove page-specific filtering for better results)
			$rows = $wpdb->get_results(
				"SELECT content,embedding,page_id 
				FROM {$wpdb->prefix}aipress_chunks",
				ARRAY_A
			);
			
			if ( empty( $rows ) ) {
				wp_send_json_success( array( 'answer' => 'I don\'t have any content available to help answer your question right now. Please contact the site administrator for assistance.' ) );
			}
			
			$good = array(); 
			
			foreach ( $rows as $row ) {
				$e = json_decode( $row['embedding'], true );
				if ( ! is_array( $e ) ) {
					continue; // Skip invalid embeddings
				}
				
				// Calculate cosine similarity
				$dot = 0;
				$ma  = 0;
				$mb  = 0;
				foreach ( $embed_q as $i => $v ) {
					$dot += $v * ( $e[ $i ] ?? 0 );
					$ma  += $v * $v;
					$mb  += ( $e[ $i ] ?? 0 ) ** 2;
				}
				$sim = $dot / ( sqrt( $ma ) * sqrt( $mb ) + 1e-9 );
				
				// Use a more reasonable similarity threshold
				if ( $sim > 0.15 ) {
					$good[] = array(
						'text' => $row['content'],
						'sim'  => $sim,
						'page_id' => $row['page_id']
					);
				}
			}
			
			// Sort by similarity and get top matches
			usort( $good, fn( $a, $b ) => $b['sim'] <=> $a['sim'] );
			$context = implode( "\n---\n", array_column( array_slice( $good, 0, 5 ), 'text' ) );

			// If no good matches, try with a lower threshold or provide helpful fallback
			if ( ! $context ) {
				// Try with lower threshold as fallback
				$fallback_good = array();
				foreach ( $rows as $row ) {
					$e = json_decode( $row['embedding'], true );
					if ( ! is_array( $e ) ) {
						continue;
					}
					
					$dot = 0;
					$ma  = 0;
					$mb  = 0;
					foreach ( $embed_q as $i => $v ) {
						$dot += $v * ( $e[ $i ] ?? 0 );
						$ma  += $v * $v;
						$mb  += ( $e[ $i ] ?? 0 ) ** 2;
					}
					$sim = $dot / ( sqrt( $ma ) * sqrt( $mb ) + 1e-9 );
					
					if ( $sim > 0.05 ) {
						$fallback_good[] = array(
							'text' => $row['content'],
							'sim'  => $sim,
							'page_id' => $row['page_id']
						);
					}
				}
				
				if ( !empty( $fallback_good ) ) {
					usort( $fallback_good, fn( $a, $b ) => $b['sim'] <=> $a['sim'] );
					$context = implode( "\n---\n", array_column( array_slice( $fallback_good, 0, 3 ), 'text' ) );
				}
			}

			// Get settings and determine which model to use
			$settings = get_option( 'aipress_settings', array() );
			$selected_model = aipress_is_pro() ? ($settings['ai_model'] ?? 'gpt-4o-mini') : 'gpt-3.5-turbo';

			// Get system prompt
			$system_prompt = $this->get_system_prompt();

			// Enhanced system prompt with better instructions
			$full_prompt = $system_prompt;
			$full_prompt .= "\n\nIMPORTANT INSTRUCTIONS:";
			$full_prompt .= "\n- Always provide helpful, specific answers when possible";
			$full_prompt .= "\n- If you have relevant information, use it to give complete answers";
			$full_prompt .= "\n- Be conversational and friendly";
			$full_prompt .= "\n- If you truly don't have the information, politely say so and suggest contacting the site for more details";
			
			if ( $context ) {
				$full_prompt .= "\n\nRelevant information from the website:\n" . $context;
				$full_prompt .= "\n\nUse the above information to answer the user's question accurately and completely.";
			} else {
				$full_prompt .= "\n\nI don't have specific information about this topic, but I should still try to be helpful and suggest they contact the site directly for detailed information.";
			}

			$resp = wp_remote_post(
				'https://api.openai.com/v1/chat/completions',
				array(
					'headers' => array(
						'Authorization' => 'Bearer ' . $this->key,
						'Content-Type'  => 'application/json',
					),
					'body'    => wp_json_encode(
						array(
							'model'      => $selected_model,
							'messages'   => array(
								array(
									'role'    => 'system',
									'content' => $full_prompt,
								),
								array(
									'role'    => 'user',
									'content' => $q,
								),
							),
							'max_tokens' => 400,
							'temperature' => 0.7,
						)
					),
					'timeout' => 30,
				)
			);
			
			if ( is_wp_error( $resp ) ) {
				wp_send_json_success( array( 'answer' => 'I\'m having trouble connecting right now. Please try again in a moment.' ) );
			}
			
			$body = wp_remote_retrieve_body( $resp );
			$data = json_decode( $body, true );
			
			// Better error handling for OpenAI response
			if ( ! isset( $data['choices'][0]['message']['content'] ) ) {
				wp_send_json_success( array( 'answer' => 'I\'m experiencing some technical difficulties. Please try again or contact support if the problem persists.' ) );
			}
			
			$ans = trim( $data['choices'][0]['message']['content'] );
			
			// Log the conversation (Pro feature)
			if ( aipress_is_pro() ) {
				aipress_log_conversation( 
					$q, 
					$ans, 
					array(
						'model' => $selected_model,
						'session_id' => isset( $_POST['session_id'] ) ? sanitize_text_field( wp_unslash( $_POST['session_id'] ) ) : ''
					)
				);
			}

			wp_send_json_success( array( 'answer' => wp_kses_post( $ans ) ) );
			
		} catch ( \Exception $e ) {
			wp_send_json_success( array( 'answer' => $e->getMessage() ) );
		}
	}

	/**
	 * Get the system prompt for the AI.
	 *
	 * @return string The system prompt to use.
	 */
	private function get_system_prompt() {
		$settings = get_option( 'aipress_settings', array() );
		
		// Check if pro and has custom prompt
		if ( aipress_is_pro() && ! empty( $settings['system_prompt'] ) ) {
			return trim( $settings['system_prompt'] );
		}
		
		// Default system prompt
		$site_name = get_bloginfo( 'name' );
		$site_description = get_bloginfo( 'description' );
		
		$prompt = "You are a helpful AI assistant for {$site_name}";
		if ( $site_description ) {
			$prompt .= " - {$site_description}";
		}
		$prompt .= ". You can ONLY answer questions about:
        - Information found on this website
        - Our products, services, and offerings
        - How to use our website
        - Company policies and procedures
        - Pricing and contact information
        
        If someone asks about topics NOT related to our website or business (like general knowledge, celebrities, other websites, cooking, weather, etc.), respond with: 'I can only help with questions about {$site_name} and our services. Is there something specific about our website I can help you with?'
        
        Base your answers strictly on the website content you've been provided.";
        
		return $prompt;
	}
	
	/**
	 * Fire OpenAI embedding endpoint.
	 *
	 * @param string $text Text to embed.
	 * @return array<int,float> Embedding vector.
	 */
	private function embed( $text ) {
		$r = wp_remote_post(
			'https://api.openai.com/v1/embeddings',
			array(
				'headers' => array(
					'Authorization' => 'Bearer ' . $this->key,
					'Content-Type'  => 'application/json',
				),
				'body'    => wp_json_encode(
					array(
						'model' => 'text-embedding-3-small',
						'input' => $text,
					)
				),
				'timeout' => 20,
			)
		);
		
		if ( is_wp_error( $r ) ) {
			return array();
		}
		
		$body = wp_remote_retrieve_body( $r );
		$data = json_decode( $body, true );
		
		if ( ! isset( $data['data'][0]['embedding'] ) ) {
			return array();
		}
		
		return $data['data'][0]['embedding'];
	}
}