<?php
/**
 * AiPress Chat — Enhanced Settings component with Pro features.
 *
 * @package aiPress_Chat
 */

namespace AIPress\Admin;

/**
 * Settings component.
 */
class Settings {
	/**
	 * Construct component.
	 */
	public function __construct() {
		add_action( 'admin_menu', array( $this, 'menu' ) );
		add_action( 'admin_init', array( $this, 'register' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_styles' ) );
		add_action( 'admin_print_styles', array( $this, 'suppress_admin_notices' ) );
		add_action( 'wp_ajax_aipress_index_now_direct', array( $this, 'ajax_index_direct' ) );
        add_action( 'wp_ajax_aipress_save_selection', array( $this, 'ajax_save_selection' ) );
        add_action( 'wp_ajax_aipress_remove_indexed_page', array( $this, 'ajax_remove_indexed_page' ) );
        add_action( 'wp_ajax_aipress_get_content_by_types', array( $this, 'ajax_get_content_by_types' ) );
	}
	
	/**
     * AJAX handler for direct indexing
     */
    public function ajax_index_direct() {
        check_ajax_referer( 'aipress_index', 'nonce' );
        
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }
        
        $content_ids = isset( $_POST['content_ids'] ) ? json_decode( wp_unslash( $_POST['content_ids'] ), true ) : array();
        
        if ( empty( $content_ids ) ) {
            wp_send_json_error( 'No content selected' );
        }
        
        // For free version, check if we would exceed the 5-page limit
        if ( ! $this->is_pro() ) {
            global $wpdb;
            
            // Get currently indexed pages
            $currently_indexed = $wpdb->get_col( "SELECT DISTINCT page_id FROM {$wpdb->prefix}aipress_chunks" );
            
            // Find which of the selected pages are new (not already indexed)
            $new_pages = array_diff( $content_ids, $currently_indexed );
            
            // Check if adding new pages would exceed the limit
            $total_after_indexing = count( $currently_indexed ) + count( $new_pages );
            
            if ( $total_after_indexing > 5 ) {
                $pages_over_limit = $total_after_indexing - 5;
                wp_send_json_error( 
                    "This would exceed the 5-page limit by {$pages_over_limit} page(s). " .
                    "Please remove some currently indexed pages first, or select fewer pages to index."
                );
            }
        }
        
        // Pass IDs directly to indexer
        $indexer = new \AIPress\Indexer();
        $result = $indexer->index_specific_content( $content_ids );
        
        wp_send_json_success( $result );
    }
    
    /**
     * Show currently indexed pages with option to remove them (Free version)
     */
    public function indexed_pages_manager() {
        global $wpdb;
        $indexed_pages = $wpdb->get_results( 
            "SELECT DISTINCT page_id FROM {$wpdb->prefix}aipress_chunks ORDER BY page_id ASC" 
        );
        
        if ( empty( $indexed_pages ) ) {
            return;
        }
        
        ?>
        <div style="margin-top: 20px; padding: 15px; background: #fff; border: 1px solid #ccd0d4; border-radius: 6px;">
            <h4 style="margin-top: 0;">Manage Indexed Pages</h4>
            <?php if ( $this->is_pro() ): ?>
                <p>These pages and posts are currently indexed. You can remove individual items if needed:</p>
            <?php else: ?>
                <p>These pages are currently indexed. To make room for new pages (5-page limit), you can remove some of these:</p>
            <?php endif; ?>
            <div id="aipress-indexed-pages">
                <?php foreach ( $indexed_pages as $row ): 
                    $page = get_post( $row->page_id );
                    if ( ! $page ) continue;
                    
                    $chunk_count = $wpdb->get_var( $wpdb->prepare(
                        "SELECT COUNT(*) FROM {$wpdb->prefix}aipress_chunks WHERE page_id = %d",
                        $row->page_id
                    ));
                ?>
                    <div class="aipress-indexed-page" style="display: flex; align-items: center; justify-content: space-between; padding: 8px 0; border-bottom: 1px solid #f0f0f0;">
                        <div>
                            <strong><?php echo esc_html( $page->post_title ); ?></strong>
                            <span style="color: #666; font-size: 13px;">(<?php echo intval( $chunk_count ); ?> chunks)</span>
                            <span style="color: #999; font-size: 12px;">- <?php echo esc_html( get_post_type_object( $page->post_type )->label ?? $page->post_type ); ?></span>
                        </div>
                        <button type="button" 
                                class="button button-secondary button-small aipress-remove-page" 
                                data-page-id="<?php echo intval( $row->page_id ); ?>"
                                data-page-title="<?php echo esc_attr( $page->post_title ); ?>">
                            Remove
                        </button>
                    </div>
                <?php endforeach; ?>
            </div>
            
            <div id="aipress-remove-status" style="margin-top: 10px;"></div>
        </div>
        
        <script>
        document.addEventListener('DOMContentLoaded', function() {
            document.querySelectorAll('.aipress-remove-page').forEach(button => {
                button.addEventListener('click', function() {
                    const pageId = this.dataset.pageId;
                    const pageTitle = this.dataset.pageTitle;
                    const statusDiv = document.getElementById('aipress-remove-status');
                    
                    if (!confirm(`Are you sure you want to remove "${pageTitle}" from indexing?`)) {
                        return;
                    }
                    
                    this.disabled = true;
                    this.textContent = 'Removing...';
                    statusDiv.innerHTML = '';
                    
                    fetch(ajaxurl, {
                        method: 'POST',
                        body: new URLSearchParams({
                            action: 'aipress_remove_indexed_page',
                            page_id: pageId,
                            nonce: '<?php echo esc_attr(wp_create_nonce( 'aipress_remove_page' )); ?>'
                        })
                    })
                    .then(r => r.json())
                    .then(r => {
                        if (r.success) {
                            statusDiv.innerHTML = '<p style="color: #46b450;">✅ Page removed successfully!</p>';
                            // Remove the page element from the DOM
                            this.closest('.aipress-indexed-page').remove();
                            // Reload after a short delay to update stats
                            setTimeout(() => location.reload(), 1500);
                        } else {
                            statusDiv.innerHTML = '<p style="color: #dc3232;">❌ Error: ' + (r.data || 'Unknown error') + '</p>';
                            this.disabled = false;
                            this.textContent = 'Remove';
                        }
                    })
                    .catch(err => {
                        statusDiv.innerHTML = '<p style="color: #dc3232;">❌ Network error: ' + err.message + '</p>';
                        this.disabled = false;
                        this.textContent = 'Remove';
                    });
                });
            });
        });
        </script>
        <?php
    }
    
    /**
     * AJAX handler to get content by selected post types
     */
    public function ajax_get_content_by_types() {
        check_ajax_referer( 'aipress_content', 'nonce' );
        
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }
        
        $post_types = isset( $_POST['post_types'] ) ? json_decode( wp_unslash( $_POST['post_types'] ), true ) : array();
        
        if ( empty( $post_types ) ) {
            wp_send_json_error( 'No post types selected' );
        }
        
        $saved_content = get_option( 'aipress_selected_content', array() );
        $html = '';
        
        foreach ( $post_types as $post_type ) {
            $posts = get_posts( array(
                'post_type' => $post_type,
                'numberposts' => 100,
                'post_status' => 'publish'
            ) );
            
            if ( ! empty( $posts ) ) {
                $post_type_obj = get_post_type_object( $post_type );
                $html .= sprintf( '<optgroup label="%s">', esc_attr( $post_type_obj->label ) );
                
                foreach ( $posts as $post ) {
                    $selected = in_array( $post->ID, $saved_content, true ) ? 'selected' : '';
                    $html .= sprintf(
                        '<option value="%s" %s>%s</option>',
                        esc_attr( $post->ID ),
                        $selected,
                        esc_html( $post->post_title ?: "(No title - ID {$post->ID})" )
                    );
                }
                
                $html .= '</optgroup>';
            }
        }
        
        if ( empty( $html ) ) {
            $html = '<option disabled>No content found for selected types</option>';
        }
        
        wp_send_json_success( array( 'html' => $html ) );
    }
    
    /**
     * AJAX handler for removing indexed pages
     */
    public function ajax_remove_indexed_page() {
        check_ajax_referer( 'aipress_remove_page', 'nonce' );
        
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }
        
        $page_id = intval( $_POST['page_id'] ?? 0 );
        
        if ( $page_id <= 0 ) {
            wp_send_json_error( 'Invalid page ID' );
        }
        
        global $wpdb;
        
        // Remove all chunks for this page
        $deleted = $wpdb->delete( 
            $wpdb->prefix . 'aipress_chunks', 
            array( 'page_id' => $page_id ), 
            array( '%d' ) 
        );
        
        if ( $deleted === false ) {
            wp_send_json_error( 'Database error occurred' );
        }
        
        // Update the selected content option to remove this page
        $selected_content = get_option( 'aipress_selected_content', array() );
        $selected_content = array_diff( $selected_content, array( $page_id ) );
        update_option( 'aipress_selected_content', $selected_content );
        
        wp_send_json_success( "Page removed from indexing. {$deleted} chunks deleted." );
    }
    
    /**
     * AJAX handler for saving selection
     */
    public function ajax_save_selection() {
        check_ajax_referer( 'aipress_save', 'nonce' );
        
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Insufficient permissions' );
        }
        
        $content_ids = isset( $_POST['content_ids'] ) ? json_decode( wp_unslash( $_POST['content_ids'] ), true ) : array();
        update_option( 'aipress_selected_content', array_map( 'intval', $content_ids ) );
        
        wp_send_json_success( 'Selection saved' );
    }
	
	/**
     * Suppress default admin notices on aiPress pages
     */
    public function suppress_admin_notices() {
        $screen = get_current_screen();
        if ( $screen && $screen->id === 'settings_page_aipress-chat' ) {
            remove_all_actions( 'admin_notices' );
            remove_all_actions( 'all_admin_notices' );
        }
    }
	
	/**
     * Enqueue admin styles
     */
    public function admin_styles( $hook ) {
        if ( $hook !== 'settings_page_aipress-chat' ) {
            return;
        }
        
        // Enqueue WordPress media uploader
        wp_enqueue_media();
        
        wp_enqueue_style(
            'aipress-admin',
            AIPRESS_URL . 'assets/admin-styles.css',
            array(),
            AIPRESS_VERSION
        );
        
        // Enqueue admin JS for media uploader
        wp_enqueue_script(
            'aipress-admin-js',
            AIPRESS_URL . 'assets/admin.js',
            array('jquery', 'media-upload'),
            AIPRESS_VERSION,
            true
        );
    }
	
	/**
	 * Check if pro features are enabled.
	 *
	 * @return bool True if pro features are enabled.
	 */
	private function is_pro() {
		return aipress_is_pro();
	}
	
	/**
	 * Menu component.
	 */
	public function menu() {
		add_options_page( 'aiPress Chat', 'aiPress Chat', 'manage_options', 'aipress-chat', array( $this, 'screen' ) );
	}
	
	/**
     * Sanitize settings component.
     *
     * @param array $input The input settings array.
     * @return array      Sanitized settings.
     */
    public function aipress_sanitize_settings( $input ) {
        // Only sanitize the main settings now (not page selection)
        
        // Sanitize OpenAI key.
        $input['openai_key'] = sanitize_text_field( $input['openai_key'] ?? '' );
    
        // Sanitize AI model
        $received_model = $input['ai_model'] ?? '';
        $allowed_models = $this->get_available_models();
        $input['ai_model'] = sanitize_text_field( $received_model );
        
        if ( ! in_array( $input['ai_model'], array_keys( $allowed_models ), true ) ) {
            $default_model = $this->is_pro() ? 'gpt-4o-mini' : 'gpt-3.5-turbo';
            $input['ai_model'] = $default_model;
        }
    
        // Sanitize system prompt (Pro only)
        if ( $this->is_pro() ) {
            $input['system_prompt'] = sanitize_textarea_field( $input['system_prompt'] ?? '' );
        } else {
            unset( $input['system_prompt'] );
        }
    
        // Sanitize branding settings (Pro only)
        if ( $this->is_pro() ) {
            $input['chat_title'] = sanitize_text_field( $input['chat_title'] ?? 'AI Assistant' );
            $input['chat_avatar'] = esc_url_raw( $input['chat_avatar'] ?? '' );
            $input['chat_icon'] = esc_url_raw( $input['chat_icon'] ?? '' );
            $input['primary_color'] = sanitize_hex_color( $input['primary_color'] ?? '#667eea' );
            $input['secondary_color'] = sanitize_hex_color( $input['secondary_color'] ?? '#764ba2' );
            $input['hide_aipress_branding'] = !empty( $input['hide_aipress_branding'] );
        } else {
            // Remove branding settings for free users
            unset( $input['chat_title'], $input['chat_avatar'], $input['chat_icon'], $input['primary_color'], $input['secondary_color'], $input['hide_aipress_branding'] );
        }
    
        return $input;
    }
    
    /**
     * Custom branding field (Pro only).
     */
    public function branding_field() {
        $chat_title = $this->opt( 'chat_title', 'AI Assistant' );
        $chat_avatar = $this->opt( 'chat_avatar', '' );
        $primary_color = $this->opt( 'primary_color', '#667eea' );
        $secondary_color = $this->opt( 'secondary_color', '#764ba2' );
        $hide_aipress_branding = $this->opt( 'hide_aipress_branding', false );
        
        echo '<div class="aipress-branding-settings">';
        
        // Chat Title
        echo '<div style="margin-bottom: 20px;">';
        echo '<label><strong>Chat Title:</strong></label><br>';
        echo '<input type="text" name="aipress_settings[chat_title]" value="' . esc_attr( $chat_title ) . '" style="width: 300px;" placeholder="AI Assistant">';
        echo '<p class="description">The title shown in the chat header.</p>';
        echo '</div>';
        
        // Chat Avatar URL
        echo '<div style="margin-bottom: 20px;">';
        echo '<label><strong>Chat Avatar URL:</strong></label><br>';
        echo '<input type="url" name="aipress_settings[chat_avatar]" value="' . esc_attr( $chat_avatar ) . '" style="width: 400px;" placeholder="https://yoursite.com/avatar.png">';
        echo '<button type="button" class="button" id="upload-avatar">Upload Image</button>';
        echo '<p class="description">Custom avatar for the AI assistant. Leave blank to use default aiPress logo.</p>';
        if ( $chat_avatar ) {
            echo '<div style="margin-top: 8px;"><img src="' . esc_url( $chat_avatar ) . '" style="width: 40px; height: 40px; border-radius: 50%; border: 2px solid #ddd;"></div>';
        }
        echo '</div>';
        
        echo '<div style="margin-bottom: 20px;">';
        echo '<label><strong>Chat Bubble Icon URL:</strong></label><br>';
        echo '<input type="url" name="aipress_settings[chat_icon]" value="' . esc_attr( $this->opt( 'chat_icon', '' ) ) . '" style="width: 400px;" placeholder="https://yoursite.com/chat-icon.png">';
        echo '<button type="button" class="button" id="upload-chat-icon">Upload Image</button>';
        echo '<p class="description">Custom icon for the floating chat bubble. Leave blank to use default chat icon. Recommended size: 60x60px.</p>';
        if ( $this->opt( 'chat_icon' ) ) {
            echo '<div style="margin-top: 8px;"><img src="' . esc_url( $this->opt( 'chat_icon' ) ) . '" style="width: 40px; height: 40px; border-radius: 50%; border: 2px solid #ddd;"></div>';
        }
        echo '</div>';

        // Primary Color
        echo '<div style="margin-bottom: 20px;">';
        echo '<label><strong>Primary Color:</strong></label><br>';
        echo '<input type="color" name="aipress_settings[primary_color]" value="' . esc_attr( $primary_color ) . '" style="width: 60px; height: 40px;">';
        echo '<span style="margin-left: 10px; color: #666;">' . esc_html( $primary_color ) . '</span>';
        echo '<p class="description">Main color for chat bubbles and buttons.</p>';
        echo '</div>';
        
        // Secondary Color
        echo '<div style="margin-bottom: 20px;">';
        echo '<label><strong>Secondary Color:</strong></label><br>';
        echo '<input type="color" name="aipress_settings[secondary_color]" value="' . esc_attr( $secondary_color ) . '" style="width: 60px; height: 40px;">';
        echo '<span style="margin-left: 10px; color: #666;">' . esc_html( $secondary_color ) . '</span>';
        echo '<p class="description">Secondary color for gradients and accents.</p>';
        echo '</div>';
        
        // Proactive Messages (in branding_field method)
        echo '<div style="margin-bottom: 20px;">';
        echo '<label><input type="checkbox" name="aipress_settings[enable_proactive]" value="1" ' . checked( $this->opt( 'enable_proactive', true ), true, false ) . '> Enable proactive chat messages</label>';
        echo '<p class="description">Show helpful messages to engage visitors who haven\'t opened the chat yet.</p>';
        echo '</div>';
        
        // Proactive Message Timing
        echo '<div style="margin-bottom: 20px;">';
        echo '<label><strong>First Message Delay (seconds):</strong></label><br>';
        echo '<input type="number" name="aipress_settings[proactive_delay]" value="' . esc_attr( $this->opt( 'proactive_delay', 10 ) ) . '" min="10" max="300" style="width: 100px;">';
        echo '<p class="description">How long to wait before showing the first proactive message.</p>';
        echo '</div>';
        
        // Hide aiPress Branding
        echo '<div style="margin-bottom: 20px;">';
        echo '<label><input type="checkbox" name="aipress_settings[hide_aipress_branding]" value="1" ' . checked( $hide_aipress_branding, true, false ) . '> Hide "Powered by aiPress" branding</label>';
        echo '<p class="description">Remove aiPress branding from the chat interface (Pro only).</p>';
        echo '</div>';
        
        echo '</div>';
        
        // Add preview
        echo '<div style="margin-top: 20px; padding: 15px; background: #f0f8ff; border-radius: 6px; border-left: 4px solid #0073aa;">';
        echo '<strong>💡 Live Preview:</strong><br>';
        echo '<p>Open the chat on your site to see your branding changes in real-time!</p>';
        echo '</div>';
        
        // Add JavaScript for media uploader
        add_action( 'admin_footer', array( $this, 'branding_upload_js' ) );
    }
    
    public function branding_upload_js() {
        ?>
        <script>
        jQuery(document).ready(function($) {
            // Avatar upload
            $('#upload-avatar').on('click', function(e) {
                e.preventDefault();
                uploadImage('aipress_settings[chat_avatar]', 'Choose Avatar Image');
            });
            
            // Chat icon upload
            $('#upload-chat-icon').on('click', function(e) {
                e.preventDefault();
                uploadImage('aipress_settings[chat_icon]', 'Choose Chat Icon');
            });
            
            function uploadImage(inputName, title) {
                var mediaUploader = wp.media({
                    title: title,
                    button: {
                        text: 'Use This Image'
                    },
                    multiple: false,
                    library: {
                        type: 'image'
                    }
                });
                
                mediaUploader.on('select', function() {
                    var attachment = mediaUploader.state().get('selection').first().toJSON();
                    var input = $('input[name="' + inputName + '"]');
                    input.val(attachment.url);
                    
                    // Show preview
                    var preview = input.nextAll('div').first();
                    if (preview.length === 0) {
                        preview = $('<div style="margin-top: 8px;"></div>');
                        input.after(preview);
                    }
                    preview.html('<img src="' + attachment.url + '" style="width: 40px; height: 40px; border-radius: 50%; border: 2px solid #ddd;">');
                });
                
                mediaUploader.open();
            }
            
            // Live color preview
            $('input[type="color"]').on('change', function() {
                $(this).next('span').text($(this).val());
            });
        });
        </script>
        <?php
    }
	
	/**
	 * Register component.
	 */
	public function register() {
		register_setting( 'aipress', 'aipress_settings', array( $this, 'aipress_sanitize_settings' ) );

		add_settings_section( 'basic', '', null, 'aipress' );

		add_settings_field( 'openai_key', __( 'OpenAI API Key', 'aipress-chat' ), array( $this, 'key_field' ), 'aipress', 'basic' );
		add_settings_field( 'ai_model', __( 'AI Model', 'aipress-chat' ), array( $this, 'model_field' ), 'aipress', 'basic' );
		
		// Pro only: Custom system prompt
		if ( $this->is_pro() ) {
			add_settings_field( 'system_prompt', __( 'Custom System Prompt (Pro)', 'aipress-chat' ), array( $this, 'system_prompt_field' ), 'aipress', 'basic' );
		}
		
		if ( $this->is_pro() ) {
            add_settings_field( 'custom_branding', __( 'Custom Branding (Pro)', 'aipress-chat' ), array( $this, 'branding_field' ), 'aipress', 'basic' );
        }
		
		// License management section
		add_settings_section(
			'aipress_license_section',
			'License Management',
			array( $this, 'license_section_callback' ),
			'aipress'
		);
		
		add_settings_field(
			'license_key',
			'License Key',
			array( $this, 'license_key_field' ),
			'aipress',
			'aipress_license_section'
		);
		
		add_settings_field(
			'license_status',
			'License Status',
			array( $this, 'license_status_field' ),
			'aipress',
			'aipress_license_section'
		);
		
	}
	
	/**
	 * Opt component.
	 *
	 * @param string $k The option key to retrieve.
	 * @param mixed  $d Default value to return if the option is not set.
	 * @return mixed    The stored option value, or `$d` if none is found.
	 */
	private function opt( string $k, $d = '' ) {
		$options = get_option( 'aipress_settings' );
		
		// Handle backward compatibility
		if ( $k === 'selected_content' && ! isset( $options['selected_content'] ) && isset( $options['selected_pages'] ) ) {
			return $options['selected_pages'];
		}
		
		return $options[ $k ] ?? $d;
	}
	
	/**
	 * OpenAI API Key field.
	 */
	public function key_field() {
		printf( '<input type="password" name="aipress_settings[openai_key]" value="%s" class="regular-text" />', esc_attr( $this->opt( 'openai_key' ) ) );
		echo '<p class="description">Your OpenAI API key for GPT responses.</p>';
	}
	
	/**
	 * Get available AI models based on version.
	 *
	 * @return array Available models.
	 */
	private function get_available_models() {
		if ( $this->is_pro() ) {
			return array(
				'gpt-4o' => 'GPT-4o (Latest & Best)',
				'gpt-4o-mini' => 'GPT-4o Mini (Fast & Efficient)',
				'gpt-4-turbo' => 'GPT-4 Turbo (Previous Latest)',
				'gpt-3.5-turbo' => 'GPT-3.5 Turbo (Budget Option)'
			);
		} else {
			return array(
				'gpt-3.5-turbo' => 'GPT-3.5 Turbo (Free Tier)'
			);
		}
	}
	
	/**
	 * Get default model based on version.
	 *
	 * @return string Default model.
	 */
	private function get_default_model() {
		return $this->is_pro() ? 'gpt-4o-mini' : 'gpt-3.5-turbo';
	}
	
	/**
	 * AI Model selection field.
	 */
	public function model_field() {
		$saved_model = $this->opt( 'ai_model', $this->get_default_model() );
		$models = $this->get_available_models();
		
		echo '<select name="aipress_settings[ai_model]" style="width: 300px;">';
		foreach ( $models as $model_id => $model_name ) {
			$selected = selected( $saved_model, $model_id, false );
			printf(
				'<option value="%s" %s>%s</option>',
				esc_attr( $model_id ),
				esc_attr( $selected ),
				esc_html( $model_name )
			);
		}
		echo '</select>';
		
		if ( $this->is_pro() ) {
			echo '<p class="description">Choose your preferred AI model. GPT-4o models provide better responses but cost more per request.</p>';
		} else {
			echo '<p class="description">Free tier includes GPT-3.5 Turbo. <strong>Upgrade to Pro</strong> for access to GPT-4o and other advanced models.</p>';
		}
		
		// Show cost information
		echo '<div style="margin-top: 10px; padding: 10px; background: #f9f9f9; border-radius: 4px; font-size: 13px;">';
		echo '<strong>Model Comparison:</strong><br>';
		if ( $this->is_pro() ) {
			echo '• <strong>GPT-4o:</strong> Best quality, highest cost (~$0.015/1k tokens)<br>';
			echo '• <strong>GPT-4o Mini:</strong> Great balance of quality & cost (~$0.0006/1k tokens)<br>';
			echo '• <strong>GPT-4 Turbo:</strong> Excellent quality (~$0.010/1k tokens)<br>';
			echo '• <strong>GPT-3.5 Turbo:</strong> Good quality, lowest cost (~$0.001/1k tokens)';
		} else {
			echo '• <strong>GPT-3.5 Turbo:</strong> Good quality for basic questions (~$0.001/1k tokens)<br>';
			echo '• <strong>GPT-4o models:</strong> Much better understanding & responses (Pro only)';
		}
		echo '</div>';
	}
	
	/**
	 * Get default system prompt.
	 *
	 * @return string Default system prompt.
	 */
	private function get_default_system_prompt() {
		$site_name = get_bloginfo( 'name' );
		$site_description = get_bloginfo( 'description' );
		
		$prompt = "You are a helpful AI assistant for {$site_name}";
		if ( $site_description ) {
			$prompt .= " - {$site_description}";
		}
		$prompt .= ". You help visitors by answering questions about the website content. Always be friendly, professional, and helpful. If you don't know something, be honest about it and suggest they contact the site owner for more information.";
		
		return $prompt;
	}
	
	/**
	 * System prompt field (Pro only).
	 */
	public function system_prompt_field() {
		$saved_prompt = $this->opt( 'system_prompt', $this->get_default_system_prompt() );
		
		echo '<div style="margin-bottom: 15px;">';
		echo '<textarea name="aipress_settings[system_prompt]" rows="6" style="width: 100%; max-width: 600px;" placeholder="' . esc_attr( $this->get_default_system_prompt() ) . '">' . esc_textarea( $saved_prompt ) . '</textarea>';
		echo '</div>';
		
		echo '<p class="description">Customize your AI\'s personality and brand voice. This system prompt tells the AI how to behave and respond to users.</p>';
		
		// Show examples and tips
		echo '<div style="margin-top: 15px; padding: 15px; background: #f0f8ff; border-radius: 6px; border-left: 4px solid #0073aa;">';
		echo '<strong>💡 Tips for great system prompts:</strong><br><br>';
		echo '<strong>Be specific about tone:</strong><br>';
		echo '<em>"Always respond in a casual, friendly tone" or "Use professional business language"</em><br><br>';
		echo '<strong>Define your brand:</strong><br>';
		echo '<em>"You represent [Company Name], a leader in web design services"</em><br><br>';
		echo '<strong>Set boundaries:</strong><br>';
		echo '<em>"If asked about pricing, direct users to contact our sales team"</em><br><br>';
		echo '<strong>Add personality:</strong><br>';
		echo '<em>"Use occasional emojis and be enthusiastic about helping customers"</em>';
		echo '</div>';
		
		// Quick templates
		echo '<div style="margin-top: 15px;">';
		echo '<strong>Quick Templates:</strong> ';
		echo '<button type="button" class="button button-small" onclick="aipress_fill_prompt(\'business\')">Professional Business</button> ';
		echo '<button type="button" class="button button-small" onclick="aipress_fill_prompt(\'casual\')">Casual & Friendly</button> ';
		echo '<button type="button" class="button button-small" onclick="aipress_fill_prompt(\'ecommerce\')">E-commerce Store</button> ';
		echo '<button type="button" class="button button-small" onclick="aipress_fill_prompt(\'default\')">Reset to Default</button>';
		echo '</div>';
		
		// Add JavaScript for templates
		add_action( 'admin_footer', array( $this, 'system_prompt_templates_js' ) );
	}
	
	/**
	 * JavaScript for system prompt templates.
	 */
	public function system_prompt_templates_js() {
		$site_name = get_bloginfo( 'name' );
		?>
		<script>
		function aipress_fill_prompt(type) {
			const textarea = document.querySelector('textarea[name="aipress_settings[system_prompt]"]');
			if (!textarea) return;
			
			const templates = {
				'business': `You are a professional AI assistant for <?php echo esc_js( $site_name ); ?>. You help visitors with their questions in a courteous, professional manner. Always maintain a business-appropriate tone, provide accurate information, and direct users to contact our team for specific inquiries about services, pricing, or appointments.`,
				
				'casual': `Hi! I'm the friendly AI assistant for <?php echo esc_js( $site_name ); ?>! 😊 I'm here to help answer your questions and make your visit awesome. I love chatting and helping people find what they need. If I don't know something, I'll be honest and point you in the right direction!`,
				
				'ecommerce': `You are a helpful shopping assistant for <?php echo esc_js( $site_name ); ?>. You help customers find products, answer questions about features and shipping, and guide them through their purchase decisions. Always be enthusiastic about our products while being honest about their benefits. For specific order issues, direct customers to contact our support team.`,
				
				'default': `<?php echo esc_js( $this->get_default_system_prompt() ); ?>`
			};
			
			if (templates[type]) {
				textarea.value = templates[type];
				textarea.focus();
			}
		}
		</script>
		<?php
	}
	
	/**
     * Content selection and indexing section (separate from main settings)
     */
    public function content_indexing_section() {
        ?>
        <div class="aipress-content-section">
            <h2>📚 Content Indexing</h2>
            <p>Select the pages you want to make available to the AI chatbot, then click "Index Now" to process them.</p>
            
            
            <form id="aipress-indexing-form" style="background: #f9f9f9; padding: 20px; border-radius: 8px; margin: 20px 0;">
                
                <?php if ( $this->is_pro() ): ?>
                    <!-- Pro: Post Types Selection -->
                    <div style="margin-bottom: 20px;">
                        <label><strong>Content Types to Include:</strong></label>
                        <?php $this->post_types_field_standalone(); ?>
                    </div>
                    
                    <!-- Pro: Content Selection -->
                    <div style="margin-bottom: 20px;">
                        <label><strong>Select Content to Index:</strong></label>
                        <?php $this->content_field_standalone(); ?>
                    </div>
                <?php else: ?>
                    <!-- Free: Pages Only -->
                    <div style="margin-bottom: 20px;">
                        <label><strong>Select Pages to Index (Max 5):</strong></label>
                        </div>
                        <?php $this->pages_field_standalone(); ?>
                    
                <?php endif; ?>
                
                <div style="display: flex; align-items: center; gap: 15px; margin-top: 20px;">
                    <?php $this->index_btn_new(); ?>
                    <div id="aipress-selection-summary" style="color: #666; font-style: italic;"></div>
                </div>
                
                <div id="aipress-index-status" style="margin-top: 15px;"></div>
            </form>
            <!-- Show stats -->
            <div style="margin-bottom: 20px;">
                <?php $this->stats_block(); ?>
                <!-- Show indexed pages manager for free version -->
                <?php $this->indexed_pages_manager(); ?>
            </div>
            
        </div>
        
        <script>
        document.addEventListener('DOMContentLoaded', function() {
            const selects = document.querySelectorAll('#aipress-indexing-form select[multiple]');
            const summary = document.getElementById('aipress-selection-summary');
            const pagesCount = document.getElementById('pages-selection-count');
            
            // Handle post type checkbox changes (Pro version)
            const postTypeCheckboxes = document.querySelectorAll('.aipress-post-type');
            const contentSelect = document.getElementById('aipress-content-select');
            
            if (postTypeCheckboxes.length > 0 && contentSelect) {
                postTypeCheckboxes.forEach(checkbox => {
                    checkbox.addEventListener('change', function() {
                        updateContentOptions();
                    });
                });
                
                // Initial load
                updateContentOptions();
            }
            
            function updateContentOptions() {
                if (!contentSelect) return;
                
                // Get selected post types
                const selectedTypes = Array.from(postTypeCheckboxes)
                    .filter(cb => cb.checked)
                    .map(cb => cb.value);
                    
                console.log('Selected post types:', selectedTypes);
                
                if (selectedTypes.length === 0) {
                    contentSelect.innerHTML = '<option disabled>Please select content types above</option>';
                    updateSummary();
                    return;
                }
                
                // Show loading
                contentSelect.innerHTML = '<option disabled>Loading content...</option>';
                
                // Fetch content via AJAX
                fetch(ajaxurl, {
                    method: 'POST',
                    body: new URLSearchParams({
                        action: 'aipress_get_content_by_types',
                        post_types: JSON.stringify(selectedTypes),
                        nonce: '<?php echo esc_attr(wp_create_nonce( 'aipress_content' )); ?>'
                    })
                })
                .then(r => r.json())
                .then(r => {
                    if (r.success) {
                        contentSelect.innerHTML = r.data.html;
                    } else {
                        contentSelect.innerHTML = '<option disabled>Error loading content: ' + (r.data || 'Unknown error') + '</option>';
                    }
                    updateSummary();
                })
                .catch(err => {
                    console.error('Error fetching content:', err);
                    contentSelect.innerHTML = '<option disabled>Network error loading content</option>';
                    updateSummary();
                });
            }
            
            function updateSummary() {
                let totalSelected = 0;
                selects.forEach(select => {
                    totalSelected += select.selectedOptions.length;
                    
                    // Enforce 5-page limit for free version
                    const maxSelections = select.getAttribute('data-max-selections');
                    if (maxSelections && totalSelected > parseInt(maxSelections)) {
                        // Remove the last selected option that exceeded the limit
                        const options = Array.from(select.selectedOptions);
                        const lastOption = options[options.length - 1];
                        lastOption.selected = false;
                        totalSelected = parseInt(maxSelections);
                        
                        // Show warning
                        if (pagesCount) {
                            pagesCount.textContent = '(Maximum 5 pages reached)';
                            pagesCount.style.color = '#d63384';
                        }
                    } else if (pagesCount && maxSelections) {
                        pagesCount.textContent = `(${totalSelected}/5 selected)`;
                        pagesCount.style.color = totalSelected >= 5 ? '#d63384' : '#666';
                    }
                });
                
                if (totalSelected === 0) {
                    summary.textContent = 'No content selected';
                } else {
                    summary.textContent = `${totalSelected} item${totalSelected !== 1 ? 's' : ''} selected`;
                }
            }
            
            selects.forEach(select => {
                select.addEventListener('change', updateSummary);
            });
            updateSummary(); // Initial update
        });
        </script>
        <?php
    }

    /**
     * Standalone post types field (not part of main settings form)
     */
    public function post_types_field_standalone() {
        $saved_types = get_option( 'aipress_selected_post_types', array( 'page' ) );
        $post_types = get_post_types( array( 'public' => true ), 'objects' );
        
        echo '<div style="margin-bottom: 10px;">';
        foreach ( $post_types as $post_type ) {
            $checked = in_array( $post_type->name, $saved_types, true );
            printf(
                '<label style="display: block; margin-bottom: 5px;"><input type="checkbox" class="aipress-post-type" value="%s" %s> %s</label>',
                esc_attr( $post_type->name ),
                checked( $checked, true, false ),
                esc_html( $post_type->label )
            );
        }
        echo '</div>';
    }
    
    /**
     * Standalone content field (not part of main settings form)
     */
    public function content_field_standalone() {
        $saved_content = get_option( 'aipress_selected_content', array() );
        $selected_types = get_option( 'aipress_selected_post_types', array( 'page' ) );
        
        echo '<select multiple size="6" id="aipress-content-select" style="width:100%; max-height: 300px;">';
        
        foreach ( $selected_types as $post_type ) {
            $posts = get_posts( array(
                'post_type' => $post_type,
                'numberposts' => 100,
                'post_status' => 'publish'
            ) );
            
            if ( ! empty( $posts ) ) {
                $post_type_obj = get_post_type_object( $post_type );
                printf( '<optgroup label="%s">', esc_attr( $post_type_obj->label ) );
                
                foreach ( $posts as $post ) {
                    printf(
                        '<option value="%s" %s>%s</option>',
                        esc_attr( $post->ID ),
                        selected( in_array( $post->ID, $saved_content, true ), true, false ),
                        esc_html( $post->post_title ?: "(No title - ID {$post->ID})" )
                    );
                }
                
                echo '</optgroup>';
            }
        }
        
        echo '</select>';
        echo '<p class="description">Hold Cmd/Ctrl to select multiple items.</p>';
    }
    
    /**
     * Standalone pages field (not part of main settings form)
     */
    public function pages_field_standalone() {
        $saved = get_option( 'aipress_selected_content', array() );
        $pages = get_pages();
        
        echo '<select multiple size="6" id="aipress-pages-select" style="width:100%">';
        foreach ( $pages as $p ) {
            printf(
                '<option value="%s" %s>%s</option>',
                esc_attr( $p->ID ),
                selected( in_array( $p->ID, $saved, true ), true, false ),
                esc_html( $p->post_title )
            );
        }
        echo '</select>';
        echo '<p class="description">Hold Cmd/Ctrl to select up to 5 pages.</p>';
    }
	
	/**
     * New index button that works independently
     */
    public function index_btn_new() {
        $saved_key = $this->opt( 'openai_key', '' );
        
        if ( empty( $saved_key ) ) {
            echo '<button type="button" class="button" disabled style="opacity:0.5;">Index Now</button>';
            echo '<span style="color: #d63384; margin-left: 10px;">⚠️ Please set your OpenAI API key first</span>';
            return;
        }
        
        echo '<button type="button" class="button button-primary" id="aipress-index-now-new">Index Selected Content</button>';
        
        add_action( 'admin_footer', array( $this, 'index_button_script' ) );
    }
    
    /**
     * JavaScript for the new index button
     */
    public function index_button_script() {
        ?>
        <script>
        document.getElementById('aipress-index-now-new')?.addEventListener('click', function() {
            const btn = this;
            const status = document.getElementById('aipress-index-status');
            
            // Get selected content
            const contentSelect = document.getElementById('aipress-content-select') || document.getElementById('aipress-pages-select');
            const selectedIds = Array.from(contentSelect.selectedOptions).map(option => option.value);
            
            if (selectedIds.length === 0) {
                status.innerHTML = '<p style="color: #d63384;">❌ Please select at least one item to index.</p>';
                return;
            }
            
            btn.disabled = true;
            btn.textContent = 'Indexing...';
            status.innerHTML = `<p style="color: #0073aa;">🔄 Indexing ${selectedIds.length} item(s)...</p>`;
            
            // Send selected IDs directly to indexer
            fetch(ajaxurl, {
                method: 'POST',
                body: new URLSearchParams({
                    action: 'aipress_index_now_direct',
                    content_ids: JSON.stringify(selectedIds),
                    nonce: '<?php echo esc_attr(wp_create_nonce( 'aipress_index' )); ?>'
                })
            })
            .then(r => r.json())
            .then(r => {
                if (r.success) {
                    status.innerHTML = '<p style="color: #46b450;">✅ ' + (r.data || 'Indexing completed!') + '</p>';
                    
                    // Save the selection for next time
                    fetch(ajaxurl, {
                        method: 'POST',
                        body: new URLSearchParams({
                            action: 'aipress_save_selection',
                            content_ids: JSON.stringify(selectedIds),
                            nonce: '<?php echo esc_attr(wp_create_nonce( 'aipress_save' )); ?>'
                        })
                    });
                    
                    // Reload stats after a short delay
                    setTimeout(() => location.reload(), 2000);
                } else {
                    status.innerHTML = '<p style="color: #dc3232;">❌ Error: ' + (r.data || 'Unknown error') + '</p>';
                }
                btn.disabled = false;
                btn.textContent = 'Index Selected Content';
            })
            .catch(err => {
                status.innerHTML = '<p style="color: #dc3232;">❌ Network error: ' + err.message + '</p>';
                btn.disabled = false;
                btn.textContent = 'Index Selected Content';
            });
        });
        </script>
        <?php
    }

	/**
	 * Stats block component.
	 */
	public function stats_block() {
		global $wpdb;
		$cnt  = (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$wpdb->prefix}aipress_chunks" );
		$last = get_option( 'aipress_last_indexed' );
		$last = $last ? date_i18n( 'Y-m-d H:i', $last ) : 'n/a';
		
		// Get number of unique pages indexed
        $pages_indexed = (int) $wpdb->get_var( "SELECT COUNT(DISTINCT page_id) FROM {$wpdb->prefix}aipress_chunks" );
        
        echo '<h3>Indexing Stats</h3><table class="widefat" style="max-width:420px"><tbody>';
        
        // Add pages indexed row
        printf(
            '<tr><td>Pages indexed</td><td><strong>%s</strong></td></tr>',
            esc_html( $pages_indexed )
        );
		printf(
			'<tr><td>Indexed chunks</td><td><strong>%s</strong></td></tr>',
			esc_html( $cnt )
		);
		printf( '<tr><td>Last indexed</td><td>%s</td></tr>', esc_html( $last ) );
		
		// Show version info
		if ( $this->is_pro() ) {
			$current_model = $this->opt( 'ai_model', $this->get_default_model() );
			echo '<tr><td>Version</td><td><span style="color: #46b450; font-weight: bold;">Pro</span></td></tr>';
			echo '<tr><td>AI Model</td><td>' . esc_html( $current_model ) . '</td></tr>';
		} else {
			$pages_indexed = (int) $wpdb->get_var( "SELECT COUNT(DISTINCT page_id) FROM {$wpdb->prefix}aipress_chunks" );
            $limit_status = $pages_indexed >= 5 ? 'style="color: #d63384; font-weight: bold;"' : '';
            echo '<tr><td>Version</td><td><span style="color: #ff9800; font-weight: bold;">Free (5 page limit)</span></td></tr>';
            echo '<tr><td>Page Limit</td><td><span ' . esc_attr( $limit_status ) . '>' . esc_attr( $pages_indexed ) . '/5 pages used</span></td></tr>';
            echo '<tr><td>AI Model</td><td>gpt-3.5-turbo</td></tr>';
		}
		
		echo '</tbody></table>';
		
		// Show which pages are currently indexed (Free version)
        if ( ! $this->is_pro() && $pages_indexed > 0 ) {
            $indexed_pages = $wpdb->get_results( 
                "SELECT DISTINCT page_id FROM {$wpdb->prefix}aipress_chunks ORDER BY page_id ASC" 
            );
            
            if ( ! empty( $indexed_pages ) ) {
                echo '<div style="margin-top: 15px; padding: 15px; background: #f0f8ff; border-radius: 6px; border-left: 4px solid #0073aa;">';
                echo '<h4 style="margin-top: 0;">Currently Indexed Pages:</h4>';
                echo '<ul style="margin: 5px 0 0 20px;">';
                
                foreach ( $indexed_pages as $row ) {
                    $page = get_post( $row->page_id );
                    if ( $page ) {
                        $chunk_count = $wpdb->get_var( $wpdb->prepare(
                            "SELECT COUNT(*) FROM {$wpdb->prefix}aipress_chunks WHERE page_id = %d",
                            $row->page_id
                        ));
                        printf(
                            '<li><strong>%s</strong> <span style="color: #666;">(%d chunks)</span></li>',
                            esc_html( $page->post_title ),
                            intval( $chunk_count )
                        );
                    }
                }
                
                echo '</ul>';
                echo '<p style="margin-bottom: 0; font-size: 13px; color: #666;"><em>To index new pages when at the 5-page limit, you\'ll need to remove some of these pages first.</em></p>';
                echo '</div>';
            }
        }
		
		// Pro: Show conversation stats
		if ( $this->is_pro() ) {
			$conv_stats = aipress_get_conversation_stats();
			if ( ! empty( $conv_stats ) ) {
				echo '<h2 style="margin-top: 25px;">Conversation Stats</h2>';
				echo '<table class="widefat" style="max-width:420px"><tbody>';
				printf( '<tr><td>Total conversations</td><td><strong>%s</strong></td></tr>', esc_html( $conv_stats['total_conversations'] ?? 0 ) );
				printf( '<tr><td>Today</td><td>%s</td></tr>', esc_html( $conv_stats['today'] ?? 0 ) );
				printf( '<tr><td>This week</td><td>%s</td></tr>', esc_html( $conv_stats['this_week'] ?? 0 ) );
				printf( '<tr><td>Unique sessions</td><td>%s</td></tr>', esc_html( $conv_stats['unique_sessions'] ?? 0 ) );
				echo '</tbody></table>';
				
				echo '<p style="margin-top: 15px;">';
				echo '<a href="' . esc_url( admin_url( 'options-general.php?page=aipress-logs' ) ) . '" class="button button-primary">View All Conversations →</a>';
				echo '</p>';
			}
		}
	}
	
	/**
	 * License section description
	 */
	public function license_section_callback() {
		echo '<p>Manage your aiPress Chat Pro license. A valid license is required to use Pro features like GPT-4 models, custom system prompts, and conversation analytics.</p>';
	}
	
	/**
	 * License key input field
	 */
	public function license_key_field() {
		$license_manager = aipress_get_license_manager();
		$license_info = $license_manager ? $license_manager->get_license_info() : array();
		$license_key = $license_info['license_key'] ?? '';
		$is_valid = ($license_info['status'] ?? '') === \AIPress\LicenseManager::STATUS_VALID;
		
		?>
		<div id="aipress-license-container">
			<?php if ( $is_valid ): ?>
				<input type="password" 
					   id="aipress_license_key" 
					   value="<?php echo esc_attr( $license_key ); ?>" 
					   readonly 
					   style="width: 400px; font-family: monospace;"
					   placeholder="••••••••••••••••••••••••••••••••">
				<button type="button" 
						id="aipress_deactivate_license" 
						class="button button-secondary"
						style="margin-left: 10px;">
					Deactivate License
				</button>
			<?php else: ?>
				<input type="text" 
					   id="aipress_license_key" 
					   value="<?php echo esc_attr( $license_key ); ?>" 
					   style="width: 400px; font-family: monospace;"
					   placeholder="Enter your Pro license key...">
				<button type="button" 
						id="aipress_activate_license" 
						class="button button-primary"
						style="margin-left: 10px;">
					Activate License
				</button>
			<?php endif; ?>
			
			<button type="button" 
					id="aipress_check_license" 
					class="button button-secondary"
					style="margin-left: 5px;">
				Check Status
			</button>
			
			<div id="aipress_license_message" style="margin-top: 10px;"></div>
		</div>
		
		<script>
		jQuery(document).ready(function($) {
			const licenseKey = $('#aipress_license_key');
			const messageDiv = $('#aipress_license_message');
			
			// Activate license
			$('#aipress_activate_license').on('click', function() {
				const btn = $(this);
				const originalText = btn.text();
				
				btn.text('Activating...').prop('disabled', true);
				messageDiv.html('');
				
				$.post(ajaxurl, {
					action: 'aipress_activate_license',
					license_key: licenseKey.val(),
					nonce: '<?php echo esc_attr( wp_create_nonce( 'aipress_license' ) ); ?>'
				})
				.done(function(response) {
					if (response.success) {
						messageDiv.html('<div style="color: #00a32a; font-weight: bold;">✅ ' + response.data.message + '</div>');
						setTimeout(() => location.reload(), 1500);
					} else {
						messageDiv.html('<div style="color: #d63384; font-weight: bold;">❌ ' + response.data + '</div>');
					}
				})
				.fail(function() {
					messageDiv.html('<div style="color: #d63384; font-weight: bold;">❌ Connection error. Please try again.</div>');
				})
				.always(function() {
					btn.text(originalText).prop('disabled', false);
				});
			});
			
			// Deactivate license
			$('#aipress_deactivate_license').on('click', function() {
				if (!confirm('Are you sure you want to deactivate your license on this site?')) {
					return;
				}
				
				const btn = $(this);
				const originalText = btn.text();
				
				btn.text('Deactivating...').prop('disabled', true);
				messageDiv.html('');
				
				$.post(ajaxurl, {
					action: 'aipress_deactivate_license',
					nonce: '<?php echo esc_attr( wp_create_nonce( 'aipress_license' ) ); ?>'
				})
				.done(function(response) {
					if (response.success) {
						messageDiv.html('<div style="color: #00a32a; font-weight: bold;">✅ ' + response.data.message + '</div>');
						setTimeout(() => location.reload(), 1500);
					} else {
						messageDiv.html('<div style="color: #d63384; font-weight: bold;">❌ ' + response.data + '</div>');
					}
				})
				.fail(function() {
					messageDiv.html('<div style="color: #d63384; font-weight: bold;">❌ Connection error. Please try again.</div>');
				})
				.always(function() {
					btn.text(originalText).prop('disabled', false);
				});
			});
			
			// Check license status
			$('#aipress_check_license').on('click', function() {
				const btn = $(this);
				const originalText = btn.text();
				
				btn.text('Checking...').prop('disabled', true);
				messageDiv.html('');
				
				$.post(ajaxurl, {
					action: 'aipress_check_license',
					nonce: '<?php echo esc_attr( wp_create_nonce( 'aipress_license' ) ); ?>'
				})
				.done(function(response) {
					if (response.success) {
						const info = response.data.info;
						let statusColor = '#d63384';
						let statusText = 'Invalid';
						
						if (info.status === 'valid') {
							statusColor = '#00a32a';
							statusText = 'Valid';
						} else if (info.status === 'expired') {
							statusText = 'Expired';
						}
						
						let message = `<div style="color: ${statusColor}; font-weight: bold;">License Status: ${statusText}</div>`;
						
						if (info.customer_name) {
							message += `<div>Licensed to: ${info.customer_name}</div>`;
						}
						
						if (info.expires && info.expires !== 'lifetime') {
							message += `<div>Expires: ${info.expires}</div>`;
						} else if (info.expires === 'lifetime') {
							message += `<div>License Type: Lifetime</div>`;
						}
						
						if (info.activations_left !== undefined) {
							message += `<div>Activations Remaining: ${info.activations_left}</div>`;
						}
						
						messageDiv.html(message);
					} else {
						messageDiv.html('<div style="color: #d63384; font-weight: bold;">❌ ' + response.data + '</div>');
					}
				})
				.fail(function() {
					messageDiv.html('<div style="color: #d63384; font-weight: bold;">❌ Connection error. Please try again.</div>');
				})
				.always(function() {
					btn.text(originalText).prop('disabled', false);
				});
			});
			
			// Enter key activation
			licenseKey.on('keypress', function(e) {
				if (e.which === 13 && $('#aipress_activate_license').is(':visible')) {
					$('#aipress_activate_license').click();
				}
			});
		});
		</script>
		
		<p class="description">
			Enter your aiPress Chat Pro license key to unlock advanced features. 
			<a href="https://aipresschat.com/pricing/" target="_blank">Don't have a license? Get one here →</a>
		</p>
		<?php
	}
	
	/**
	 * License status display field
	 */
	public function license_status_field() {
		$license_manager = aipress_get_license_manager();
		
		if ( ! $license_manager ) {
			echo '<div style="color: #d63384;">License manager not available.</div>';
			return;
		}
		
		$license_info = $license_manager->get_license_info();
		$status = $license_info['status'] ?? 'unknown';
		$license_key = $license_info['license_key'] ?? '';
		
		// Status colors and messages
		$status_config = array(
			'valid' => array(
				'color' => '#00a32a',
				'icon' => '✅',
				'message' => 'Active - Pro features enabled'
			),
			'expired' => array(
				'color' => '#d63384',
				'icon' => '⏰',
				'message' => 'Expired - Please renew your license'
			),
			'invalid' => array(
				'color' => '#d63384',
				'icon' => '❌',
				'message' => 'Invalid license key'
			),
			'site_inactive' => array(
				'color' => '#d63384',
				'icon' => '🚫',
				'message' => 'Not activated for this site'
			),
			'inactive' => array(
				'color' => '#d63384',
				'icon' => '⚠️',
				'message' => 'License inactive'
			),
			'disabled' => array(
				'color' => '#d63384',
				'icon' => '🔒',
				'message' => 'License disabled'
			),
			'unknown' => array(
				'color' => '#787c82',
				'icon' => '❓',
				'message' => 'Status unknown'
			)
		);
		
		if ( empty( $license_key ) ) {
			echo '<div style="color: #787c82; font-weight: bold;">No license key entered</div>';
			echo '<p class="description">Enter a license key above to activate Pro features.</p>';
			return;
		}
		
		$config = $status_config[$status] ?? $status_config['unknown'];
		
		echo '<div style="color: ' . esc_attr( $config['color'] ) . '; font-weight: bold; font-size: 14px;">';
		echo esc_html( $config['icon'] ) . ' ' . esc_html( $config['message'] );
		echo '</div>';
		
		// Additional info for valid licenses
		if ( $status === 'valid' ) {
			echo '<div style="margin-top: 8px; font-size: 13px; color: #50575e;">';
			
			if ( ! empty( $license_info['customer_name'] ) ) {
				echo '<div><strong>Licensed to:</strong> ' . esc_html( $license_info['customer_name'] ) . '</div>';
			}
			
			if ( ! empty( $license_info['expires'] ) ) {
				if ( $license_info['expires'] === 'lifetime' ) {
					echo '<div><strong>License Type:</strong> Lifetime</div>';
				} else {
					echo '<div><strong>Expires:</strong> ' . esc_html( gmdate( 'F j, Y', strtotime( $license_info['expires'] ) ) ) . '</div>';
				}
			}
			
			if ( isset( $license_info['activations_left'] ) ) {
				echo '<div><strong>Activations Remaining:</strong> ' . intval( $license_info['activations_left'] ) . '</div>';
			}
			
			if ( ! empty( $license_info['last_check'] ) ) {
				echo '<div><strong>Last Verified:</strong> ' . esc_html( human_time_diff( $license_info['last_check'] ) ) . ' ago</div>';
			}
			
			echo '</div>';
		}
		
		// Help text
		if ( $status !== 'valid' ) {
			echo '<p class="description">';
			echo 'Having issues? <a href="mailto:contact@markh.design">Contact support</a> for assistance.';
			echo '</p>';
		}
	}
	
	/**
     * Simple upgrade teaser above save button
     */
    private function simple_upgrade_teaser() {
        ?>
        <div class="aipress-simple-upgrade">
            <div class="aipress-simple-upgrade-content">
                <div class="aipress-simple-upgrade-text">
                    <h3>🚀 Ready for More?</h3>
                    <p>Unlock unlimited pages, GPT-4 models, custom prompts, conversation analytics, and more with aiPress Chat Pro.</p>
                </div>
                <div class="aipress-simple-upgrade-action">
                    <a href="https://aipresschat.com/pricing/" target="_blank" class="button button-primary aipress-simple-upgrade-btn">
                        Compare Plans & Upgrade →
                    </a>
                </div>
            </div>
        </div>
        <?php
    }
	
	/**
	 * Screen component.
	 */
	public function screen() {
        ?>
        <div class="wrap">
            <div class="aipress-admin-header">
                <div class="aipress-header-left">
                    <img
                        src="<?php echo esc_url( AIPRESS_URL . 'assets/logo-aipress.png' ); ?>"
                        alt="<?php esc_attr_e( 'aiPress Chat', 'aipress-chat' ); ?>"
                        class="aipress-logo"
                    />
                    <h1 class="aipress-title">
                        <?php esc_html_e( 'aiPress Chat', 'aipress-chat' ); ?>
                        <?php if ( $this->is_pro() ) : ?>
                            <span class="aipress-pro-badge">PRO</span>
                        <?php endif; ?>
                    </h1>
                </div>
                <div class="aipress-header-right">
                    <img
                            src="<?php echo esc_url( AIPRESS_URL . 'assets/logo-markh.png' ); ?>"
                            alt="<?php esc_attr_e( 'MarkH Design', 'aipress-chat' ); ?>"
                            class="aipress-donate-logo"
                        />
                    <a
                        href="https://ko-fi.com/markhdesign"
                        target="_blank"
                        class="button button-primary aipress-donate-btn"
                    >
                        
                        <?php esc_html_e( 'Donate to MarkH Design', 'aipress-chat' ); ?>
                    </a>
                </div>
            </div>
            
            <form method="post" action="options.php">
            
                <?php
                if ( ! $this->is_pro() ) {
                    $this->simple_upgrade_teaser();
                }
                settings_fields( 'aipress' );
                ?>
                
                <!-- AI SETTINGS SECTION -->
                <div class="aipress-section aipress-ai-settings">
                    <h2 class="aipress-section-title">⚙️ AI Settings</h2>
                    <div class="aipress-section-content">
                        <?php 
                        echo '<table class="form-table">';
                        do_settings_fields( 'aipress', 'basic' ); 
                        echo '</table>';
                        ?>
                    </div>
                    
                <?php submit_button(); ?>
                </div>
                
            </form>
            
            <!-- CONTENT INDEXING SECTION -->
            <div class="aipress-section aipress-content-indexing">
                <div class="aipress-section-content">
                    <?php $this->content_indexing_section(); ?>
                </div>
            </div>
            
            <!-- LICENSE MANAGEMENT SECTION -->
            <div class="aipress-section aipress-license-management">
                <h2 class="aipress-section-title">🔑 License Management</h2>
                <div class="aipress-section-content">
                    <?php
                    $this->license_section_callback();
                    echo '<table class="form-table">';
                    do_settings_fields( 'aipress', 'aipress_license_section' );
                    echo '</table>';
                    ?>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Display system messages in organized layout
     */
    private function show_system_messages() {
        // Only show messages on the aiPress settings page, and only if they haven't been shown already
        $screen = get_current_screen();
        if ( ! $screen || $screen->id !== 'settings_page_aipress-chat' ) {
            return;
        }
    }
}