<?php

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
/**
 * Plugin Name: aiPress Chat
 * Plugin URI: https://aipresschat.com/
 * Description: Instant AI Support for WordPress. Add a ChatGPT-powered chat bubble to answer visitor questions using your website content.
 * Version: 1.2.0
 * Author: MarkH Design
 * Author URI: https://markh.design
 * Text Domain: aipress-chat
 * Requires at least: 6.2
 * Tested up to: 6.8
 * Requires PHP: 7.4
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Plugin constants
define( 'AIPRESS_VERSION', '1.2.0' );
define( 'AIPRESS_PLUGIN_FILE', __FILE__ );
define( 'AIPRESS_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'AIPRESS_URL', plugin_dir_url( __FILE__ ) );

// Include required files
require_once AIPRESS_PLUGIN_DIR . 'includes/class-license-manager.php';
require_once AIPRESS_PLUGIN_DIR . 'includes/class-chat.php';
require_once AIPRESS_PLUGIN_DIR . 'includes/class-settings.php';
require_once AIPRESS_PLUGIN_DIR . 'includes/class-indexer.php';
require_once AIPRESS_PLUGIN_DIR . 'includes/class-conversation-logs.php';
require_once AIPRESS_PLUGIN_DIR . 'includes/class-meta-box.php';

// Global instances
global $aipress_license_manager;

/**
 * Initialize the plugin
 */
function aipress_init() {
	global $aipress_license_manager;
	
	// Initialize license manager first
	$aipress_license_manager = new \AIPress\LicenseManager();
	
	// Initialize other components
	new \AIPress\Chat();
	new \AIPress\Admin\Settings();
	new \AIPress\Indexer();
	
	// Only initialize conversation logs if Pro
	if ( aipress_is_pro() ) {
		new \AIPress\Admin\ConversationLogs();
	    new \AIPress\Admin\MetaBox();
	}
}
add_action( 'plugins_loaded', 'aipress_init' );

/**
 * Check if Pro features are available
 *
 * @return bool
 */
function aipress_is_pro() {
	global $aipress_license_manager;
	
	// Fallback to constant for development/testing
	if ( defined( 'AIPRESS_PRO' ) && AIPRESS_PRO ) {
		return true;
	}
	
	// Check license manager
	if ( $aipress_license_manager && method_exists( $aipress_license_manager, 'is_licensed' ) ) {
		return $aipress_license_manager->is_licensed();
	}
	
	return false;
}

/**
 * Get license manager instance
 *
 * @return \AIPress\LicenseManager|null
 */
function aipress_get_license_manager() {
	global $aipress_license_manager;
	return $aipress_license_manager;
}

/**
 * Get conversation statistics (Pro feature)
 *
 * @return array
 */
function aipress_get_conversation_stats() {
	if ( ! aipress_is_pro() ) {
		return array();
	}
	
	global $wpdb;
	$table = $wpdb->prefix . 'aipress_logs';
	
	// Check if table exists
	if ( $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table ) ) !== $table ) {
		return array();
	}
	
	$today = current_time( 'Y-m-d' );
	$week_start = gmdate( 'Y-m-d', strtotime( 'monday this week' ) );
	
	$stats = array();
	
	// Total conversations
    $stats['total_conversations'] = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM %i", $table ) );
	
	// Today's conversations
    $stats['today'] = (int) $wpdb->get_var( $wpdb->prepare(
        "SELECT COUNT(*) FROM %i WHERE DATE(created_at) = %s",
        $table,
        $today
    ) );
    
    // This week's conversations
    $stats['this_week'] = (int) $wpdb->get_var( $wpdb->prepare(
        "SELECT COUNT(*) FROM %i WHERE DATE(created_at) >= %s",
        $table,
        $week_start
    ) );
	
	// Unique sessions
	$stats['unique_sessions'] = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(DISTINCT session_id) FROM %i", $table ) );
	
	return $stats;
}

/**
 * Plugin activation hook
 */
function aipress_activate() {
	// Create database tables
	aipress_create_tables();
	
	// Set default options
	$default_settings = array(
		'openai_key' => '',
		'ai_model' => 'gpt-3.5-turbo',
		'system_prompt' => '',
		'selected_content' => array(),
		'post_types' => array( 'page' ),
	);
	
	add_option( 'aipress_settings', $default_settings );
	
	// Flush rewrite rules
	flush_rewrite_rules();
}
register_activation_hook( __FILE__, 'aipress_activate' );

/**
 * Plugin deactivation hook
 */
function aipress_deactivate() {
	// Clear scheduled events
	wp_clear_scheduled_hook( 'aipress_daily_license_check' );
	
	// Flush rewrite rules
	flush_rewrite_rules();
}
register_deactivation_hook( __FILE__, 'aipress_deactivate' );

/**
 * Create database tables
 */
function aipress_create_tables() {
	global $wpdb;
	
	$charset_collate = $wpdb->get_charset_collate();
	
	// Chunks table
    $chunks_table = $wpdb->prefix . 'aipress_chunks';
    $chunks_sql = "CREATE TABLE $chunks_table (
        id bigint(20) NOT NULL AUTO_INCREMENT,
        page_id bigint(20) NOT NULL,
        chunk_index int NOT NULL DEFAULT 0,
        content text NOT NULL,
        embedding longtext NOT NULL,
        created_at datetime DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        KEY page_id (page_id)
    ) $charset_collate;";
	
	// Logs table
	$logs_table = $wpdb->prefix . 'aipress_logs';
	$logs_sql = "CREATE TABLE $logs_table (
		id bigint(20) NOT NULL AUTO_INCREMENT,
		session_id varchar(100) NOT NULL,
		user_message text NOT NULL,
		ai_response text NOT NULL,
		page_id bigint(20) DEFAULT NULL,
		user_ip varchar(45) DEFAULT NULL,
		user_agent text DEFAULT NULL,
		model varchar(50) DEFAULT NULL,
		response_time int DEFAULT NULL,
		created_at datetime DEFAULT CURRENT_TIMESTAMP,
		PRIMARY KEY (id),
		KEY session_id (session_id),
		KEY page_id (page_id),
		KEY created_at (created_at)
	) $charset_collate;";
	
	require_once ABSPATH . 'wp-admin/includes/upgrade.php';
	dbDelta( $chunks_sql );
	dbDelta( $logs_sql );
}

/**
 * Check and update database schema if needed
 */
function aipress_check_database_schema() {
    global $wpdb;
    $table = $wpdb->prefix . 'aipress_chunks';
    
    // Check if chunk_index column exists
    $columns = $wpdb->get_results("SHOW COLUMNS FROM {$table} LIKE 'chunk_index'");
    
    if (empty($columns)) {
        // Add the missing chunk_index column for existing installations
        $wpdb->query("ALTER TABLE {$table} ADD COLUMN chunk_index INT NOT NULL DEFAULT 0 AFTER page_id");
    }
}

// Run schema check on admin init
add_action('admin_init', 'aipress_check_database_schema');

/**
 * Log conversation (Pro feature)
 *
 * @param string $user_message
 * @param string $ai_response  
 * @param array $metadata
 */
function aipress_log_conversation( $user_message, $ai_response, $metadata = array() ) {
	if ( ! aipress_is_pro() ) {
		return;
	}
	
	global $wpdb;
	
	$wpdb->insert(
		$wpdb->prefix . 'aipress_logs',
		array(
			'session_id' => $metadata['session_id'] ?? '',
			'user_message' => $user_message,
			'ai_response' => $ai_response,
			'page_id' => $metadata['page_id'] ?? get_queried_object_id(),
			'user_ip' => $metadata['user_ip'] ?? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ?? '' ) ),
			'user_agent' => $metadata['user_agent'] ?? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ?? '' ) ),
			'model' => $metadata['model'] ?? 'gpt-3.5-turbo',
			'response_time' => $metadata['response_time'] ?? null,
			'created_at' => current_time( 'mysql' ),
		),
		array( '%s', '%s', '%s', '%d', '%s', '%s', '%s', '%d', '%s' )
	);
}



/**
 * Add settings link to plugin page
 */
function aipress_plugin_action_links( $links ) {
	$settings_link = '<a href="' . admin_url( 'options-general.php?page=aipress-chat' ) . '">Settings</a>';
	array_unshift( $links, $settings_link );
	
	if ( ! aipress_is_pro() ) {
		$pro_link = '<a href="https://aipresschat.com/pricing/" target="_blank" style="color: #d63384; font-weight: bold;">Upgrade to Pro</a>';
		array_unshift( $links, $pro_link );
	}
	
	return $links;
}
add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), 'aipress_plugin_action_links' );

// Handle AJAX request for conversation details
add_action('wp_ajax_get_conversation_details', 'handle_get_conversation_details');

function handle_get_conversation_details() {
    if (!current_user_can('manage_options')) {
        wp_die('Unauthorized');
    }
    
    $conversation_id = intval($_POST['conversation_id']);
    
    global $wpdb;
    $table_name = $wpdb->prefix . 'aipress_logs'; // Changed from aipress_conversations
    
    // Get the conversation
    $conversation = $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM $table_name WHERE id = %d",
        $conversation_id
    ));
    
    if (!$conversation) {
        wp_send_json_error('Conversation not found');
        return;
    }
    
    // Since your table structure is different, display the data differently
    $html = '';
    $html .= '<div class="chat-message user">';
    $html .= '<div class="chat-message-role">User:</div>';
    $html .= '<div class="chat-message-content">' . nl2br(esc_html($conversation->user_message)) . '</div>';
    $html .= '</div>';
    
    $html .= '<div class="chat-message assistant">';
    $html .= '<div class="chat-message-role">Assistant:</div>';
    $html .= '<div class="chat-message-content">' . nl2br(esc_html($conversation->ai_response)) . '</div>';
    $html .= '</div>';
    
    // Add metadata
    $html .= '<div style="margin-top: 20px; padding: 10px; background: #f5f5f5; border-radius: 4px;">';
    $html .= '<strong>Conversation Details:</strong><br>';
    $html .= 'Session ID: ' . esc_html($conversation->session_id) . '<br>';
    $html .= 'User IP: ' . esc_html($conversation->user_ip) . '<br>';
    $html .= 'Model: ' . esc_html($conversation->model) . '<br>';
    if ($conversation->response_time) {
        $html .= 'Response Time: ' . esc_html($conversation->response_time) . 'ms<br>';
    }
    $html .= 'Date: ' . esc_html(date('F j, Y g:i A', strtotime($conversation->created_at)));
    $html .= '</div>';
    
    wp_send_json_success($html);
}