<?php
/**
 * AiPress Chat — License Management CLIENT PLUGIN
 *
 * FIXED VERSION to reduce excessive requests
 */

namespace AIPress;

class LicenseManager {
	
	const LICENSE_SERVER = 'https://aipresschat.com/wp-json/aipress-licensing/v1';
	
	const STATUS_VALID = 'valid';
	const STATUS_INVALID = 'invalid';
	const STATUS_EXPIRED = 'expired';
	const STATUS_INACTIVE = 'inactive';
	const STATUS_SITE_INACTIVE = 'site_inactive';
	const STATUS_DISABLED = 'disabled';
	
	public function __construct() {
		add_action( 'wp_ajax_aipress_activate_license', array( $this, 'ajax_activate_license' ) );
		add_action( 'wp_ajax_aipress_deactivate_license', array( $this, 'ajax_deactivate_license' ) );
		add_action( 'wp_ajax_aipress_check_license', array( $this, 'ajax_check_license' ) );
		
		// FIXED: Only schedule once and make it weekly instead of daily
		if ( ! wp_next_scheduled( 'aipress_weekly_license_check' ) ) {
			// Clear any old daily checks first
			wp_clear_scheduled_hook( 'aipress_daily_license_check' );
			
			// Schedule weekly check instead of daily
			wp_schedule_event( time(), 'weekly', 'aipress_weekly_license_check' );
		}
		add_action( 'aipress_weekly_license_check', array( $this, 'weekly_license_check' ) );
	}
	
	/**
	 * FIXED: Check if the current license is valid with better caching
	 */
	public function is_licensed() {
		$license_data = get_option( 'aipress_license_data', array() );
		
		// FIXED: Cache valid licenses for 7 days instead of 1 day
		if ( 
			isset( $license_data['status'] ) && 
			$license_data['status'] === self::STATUS_VALID &&
			isset( $license_data['expires'] ) &&
			( $license_data['expires'] === 'lifetime' || strtotime( $license_data['expires'] ) > time() ) &&
			isset( $license_data['last_check'] ) &&
			( time() - $license_data['last_check'] ) < (7 * DAY_IN_SECONDS) // 7 days cache
		) {
			return true;
		}
		
		// FIXED: Only check remotely if we haven't checked in the last 24 hours
		$license_key = get_option( 'aipress_license_key', '' );
		if ( empty( $license_key ) ) {
			return false;
		}
		
		// FIXED: Add transient to prevent multiple remote checks
		$last_remote_check = get_transient( 'aipress_last_remote_check' );
		if ( $last_remote_check ) {
			// We've checked recently, use cached result
			return isset( $license_data['status'] ) && $license_data['status'] === self::STATUS_VALID;
		}
		
		// Perform remote check (non-blocking)
		$this->check_license_remote( $license_key, false );
		
		// Set transient to prevent another check for 24 hours
		set_transient( 'aipress_last_remote_check', time(), DAY_IN_SECONDS );
		
		// Re-check after potential remote validation
		$license_data = get_option( 'aipress_license_data', array() );
		return isset( $license_data['status'] ) && $license_data['status'] === self::STATUS_VALID;
	}
	
	/**
	 * Get license status information
	 */
	public function get_license_info() {
		$license_key = get_option( 'aipress_license_key', '' );
		$license_data = get_option( 'aipress_license_data', array() );
		
		return array(
			'license_key' => $license_key,
			'status' => $license_data['status'] ?? 'unknown',
			'expires' => $license_data['expires'] ?? null,
			'customer_name' => $license_data['customer_name'] ?? '',
			'activations_left' => $license_data['activations_left'] ?? 0,
			'last_check' => $license_data['last_check'] ?? 0,
		);
	}
	
	/**
	 * Activate a license key
	 */
	public function activate_license( $license_key ) {
		$license_key = sanitize_text_field( trim( $license_key ) );
		
		if ( empty( $license_key ) ) {
			return array(
				'success' => false,
				'message' => 'Please enter a license key.'
			);
		}
		
		$response = $this->api_request( 'activate_license', array(
			'license' => $license_key,
			'url' => home_url(),
			'site_name' => get_bloginfo( 'name' ),
		) );
		
		if ( is_wp_error( $response ) ) {
			return array(
				'success' => false,
				'message' => 'Connection error: ' . $response->get_error_message()
			);
		}
		
		$response_code = wp_remote_retrieve_response_code( $response );
		$body = wp_remote_retrieve_body( $response );
		
		if ( $response_code !== 200 ) {
			return array(
				'success' => false,
				'message' => 'Server error (HTTP ' . $response_code . '): ' . $body
			);
		}
		
		$data = json_decode( $body, true );
		
		if ( ! $data ) {
			return array(
				'success' => false,
				'message' => 'Invalid response from license server.'
			);
		}
		
		if ( isset( $data['license'] ) && $data['license'] === self::STATUS_VALID ) {
			// Store license data
			update_option( 'aipress_license_key', $license_key );
			update_option( 'aipress_license_data', array(
				'status' => $data['license'],
				'expires' => $data['expires'] ?? 'lifetime',
				'customer_name' => $data['customer_name'] ?? '',
				'activations_left' => $data['activations_left'] ?? 0,
				'last_check' => time(),
			) );
			
			// FIXED: Clear the remote check transient so we can verify immediately
			delete_transient( 'aipress_last_remote_check' );
			
			return array(
				'success' => true,
				'message' => 'License activated successfully!',
				'data' => $data
			);
		} else {
			// Store failed status
			update_option( 'aipress_license_data', array(
				'status' => $data['license'] ?? self::STATUS_INVALID,
				'last_check' => time(),
			) );
			
			$message = $this->get_error_message( $data['license'] ?? self::STATUS_INVALID );
			return array(
				'success' => false,
				'message' => $message
			);
		}
	}
	
	/**
	 * Deactivate current license
	 */
	public function deactivate_license() {
		$license_key = get_option( 'aipress_license_key', '' );
		
		if ( empty( $license_key ) ) {
			return array(
				'success' => false,
				'message' => 'No license key found to deactivate.'
			);
		}
		
		$response = $this->api_request( 'deactivate_license', array(
			'license' => $license_key,
			'url' => home_url(),
		) );
		
		// Clear local license data regardless of API response
		delete_option( 'aipress_license_key' );
		delete_option( 'aipress_license_data' );
		delete_transient( 'aipress_last_remote_check' );
		
		if ( is_wp_error( $response ) ) {
			return array(
				'success' => true, // Still success since we cleared locally
				'message' => 'License deactivated locally. Connection error: ' . $response->get_error_message()
			);
		}
		
		return array(
			'success' => true,
			'message' => 'License deactivated successfully!'
		);
	}
	
	/**
	 * FIXED: Check license status remotely with better error handling
	 */
	private function check_license_remote( $license_key, $blocking = true ) {
		// FIXED: Don't check if we're in maintenance mode or doing updates
		if ( defined( 'WP_INSTALLING' ) || wp_installing() ) {
			return false;
		}
		
		$response = $this->api_request( 'check_license', array(
			'license' => $license_key,
			'url' => home_url(),
		), $blocking );
		
		if ( is_wp_error( $response ) || ! $blocking ) {
			return false;
		}
		
		$response_code = wp_remote_retrieve_response_code( $response );
		if ( $response_code !== 200 ) {
			return false;
		}
		
		$body = wp_remote_retrieve_body( $response );
		$data = json_decode( $body, true );
		
		if ( $data && isset( $data['license'] ) ) {
			update_option( 'aipress_license_data', array(
				'status' => $data['license'],
				'expires' => $data['expires'] ?? null,
				'customer_name' => $data['customer_name'] ?? '',
				'activations_left' => $data['activations_left'] ?? 0,
				'last_check' => time(),
			) );
			
			return $data['license'] === self::STATUS_VALID;
		}
		
		return false;
	}
	
	/**
	 * FIXED: Make API request with better timeout and retry logic
	 */
	private function api_request( $action, $data = array(), $blocking = true ) {
		$api_url = self::LICENSE_SERVER . '/' . $action;
		
		$request_data = array_merge( $data, array(
			'item_id' => 1,
			'version' => AIPRESS_VERSION,
		) );
		
		$timeout = $blocking ? 10 : 5; // Reduced timeout
		
		$args = array(
			'method' => 'POST',
			'timeout' => $timeout,
			'blocking' => $blocking,
			'body' => $request_data,
			'headers' => array(
				'User-Agent' => 'AIPress/' . AIPRESS_VERSION . '; ' . home_url(),
			),
			'sslverify' => true, // Always verify SSL
		);
		
		return wp_remote_request( $api_url, $args );
	}
	
	/**
	 * Get human-readable error message for license status
	 */
	private function get_error_message( $status ) {
		$messages = array(
			self::STATUS_INVALID => 'Invalid license key. Please check your key and try again.',
			self::STATUS_EXPIRED => 'Your license has expired. Please renew to continue using Pro features.',
			self::STATUS_INACTIVE => 'License key is inactive. Please contact support.',
			self::STATUS_SITE_INACTIVE => 'License key is not activated for this site. Each license can be used on a limited number of sites.',
			self::STATUS_DISABLED => 'License key has been disabled. Please contact support.',
		);
		
		return $messages[ $status ] ?? 'Unknown license error. Please contact support.';
	}
	
	/**
	 * FIXED: Weekly license check (was daily)
	 */
	public function weekly_license_check() {
		$license_key = get_option( 'aipress_license_key', '' );
		if ( ! empty( $license_key ) ) {
			// FIXED: Add additional throttling for background checks
			$last_background_check = get_transient( 'aipress_last_background_check' );
			if ( $last_background_check ) {
				return; // Skip if we've done a background check recently
			}
			
			$this->check_license_remote( $license_key, false );
			
			// Throttle background checks to once per 3 days
			set_transient( 'aipress_last_background_check', time(), 3 * DAY_IN_SECONDS );
		}
	}
	
	/**
	 * AJAX handler for license activation
	 */
	public function ajax_activate_license() {
		check_ajax_referer( 'aipress_license', 'nonce' );
		
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( 'Insufficient permissions.' );
		}
		
		$license_key = isset( $_POST['license_key'] ) ? sanitize_text_field( wp_unslash( $_POST['license_key'] ) ) : '';
		$result = $this->activate_license( $license_key );
		
		if ( $result['success'] ) {
			wp_send_json_success( $result );
		} else {
			wp_send_json_error( $result['message'] );
		}
	}
	
	/**
	 * AJAX handler for license deactivation
	 */
	public function ajax_deactivate_license() {
		check_ajax_referer( 'aipress_license', 'nonce' );
		
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( 'Insufficient permissions.' );
		}
		
		$result = $this->deactivate_license();
		wp_send_json_success( $result );
	}
	
	/**
	 * AJAX handler for license status check
	 */
	public function ajax_check_license() {
		check_ajax_referer( 'aipress_license', 'nonce' );
		
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( 'Insufficient permissions.' );
		}
		
		$license_key = get_option( 'aipress_license_key', '' );
		if ( empty( $license_key ) ) {
			wp_send_json_error( 'No license key found.' );
		}
		
		$result = $this->check_license_remote( $license_key, true );
		$license_info = $this->get_license_info();
		
		wp_send_json_success( array(
			'valid' => $result,
			'info' => $license_info
		) );
	}
}
?>